<?php
// Fichier : ASE_Maintenance/generer_bon_livraison.php

session_start();
// Inclure le fichier de configuration de la base de données.
// Assurez-vous que 'php/config.php' est le chemin correct depuis ce fichier.
require_once 'php/config.php';

// Vérifier si l'utilisateur est connecté et s'il a bien le rôle de Magasinier.
// Si ce n'est pas le cas, rediriger vers la page de connexion.
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || $_SESSION['role'] !== 'Magasinier') {
    header('Location: index.php'); // Redirection vers la page de connexion à la racine du projet
    exit();
}

// --- DÉBUT DE LA LOGIQUE DE VÉRIFICATION ET RÉCUPÉRATION DES DONNÉES ---

// Vérifier si l'ID de commande est présent dans l'URL et s'il est numérique.
// C'est le point de défaillance principal si l'erreur "ID manquant ou invalide" apparaît.
if (!isset($_GET['id_commande']) || !is_numeric($_GET['id_commande'])) {
    // Message de débogage pour comprendre pourquoi l'ID est manquant/invalide
    error_log("Tentative d'accès à generer_bon_livraison.php sans ID de commande valide.");
    error_log("Contenu de \$_GET: " . print_r($_GET, true)); // Log les données GET reçues

    die("ID de commande manquant ou invalide pour la génération du Bon de Livraison. Veuillez revenir à la page de réception et réessayer.");
}

// Convertir l'ID de commande en entier pour des raisons de sécurité et de type.
$orderId = (int)$_GET['id_commande'];

try {
    // Obtenir la connexion PDO à la base de données.
    $pdo = getDbConnection();

    // Récupérer les informations principales de la commande d'approvisionnement.
    // Joindre avec la table 'utilisateurs' pour obtenir le nom du magasinier.
    $stmtOrder = $pdo->prepare("SELECT co.id_commande, co.date_commande, co.statut_commande, co.statut_livraison,
                                     u.nom_utilisateur AS magasinier_nom
                                FROM commandes_appro co
                                JOIN utilisateurs u ON co.id_magasinier = u.id_utilisateur
                                WHERE co.id_commande = :order_id");
    $stmtOrder->execute([':order_id' => $orderId]);
    $order = $stmtOrder->fetch(PDO::FETCH_ASSOC);

    // Si la commande n'est pas trouvée, afficher une erreur.
    if (!$order) {
        die("Commande introuvable avec l'ID : " . htmlspecialchars($orderId) . ". Vérifiez l'ID et réessayez.");
    }

    // Récupérer les détails des pièces associées à cette commande.
    // Le Bon de Livraison doit uniquement lister les pièces qui ont été EFFECTIVEMENT reçues (quantite_recue > 0).
    // Correction: 'pieces' a été remplacé par 'piece' pour le nom de la table.
    $stmtDetails = $pdo->prepare("SELECT dc.id_detail, dc.id_piece, dc.quantite_commandee, dc.quantite_recue, dc.prix_ht_unitaire, dc.tva_unitaire,
                                         p.nom_piece, p.reference, p.code_piece
                                  FROM details_commande_appro dc
                                  JOIN piece p ON dc.id_piece = p.id_piece
                                  WHERE dc.id_commande = :order_id AND dc.quantite_recue > 0");
    $stmtDetails->execute([':order_id' => $orderId]);
    $orderDetails = $stmtDetails->fetchAll(PDO::FETCH_ASSOC);

    // Si aucune pièce n'a été reçue pour cette commande, impossible de générer le BDL.
    if (empty($orderDetails)) {
        die("Aucune pièce n'a encore été reçue pour la commande N° " . htmlspecialchars($orderId) . " (ou toutes les réceptions ont été annulées). Impossible de générer un bon de livraison pour cette commande.");
    }

    // Formatter l'ID du Bon de Livraison (par exemple, BL0001, BL1234).
    $formattedDeliveryId = 'BL' . str_pad($orderId, 4, '0', STR_PAD_LEFT);

} catch (PDOException $e) {
    // Gérer les erreurs de base de données.
    error_log("Erreur de base de données lors de la récupération des informations de commande pour BL (ID: " . $orderId . ") : " . $e->getMessage());
    die("Erreur interne du serveur lors du chargement des détails de la commande. Veuillez contacter l'administrateur.");
} catch (Exception $e) {
    // Gérer toute autre erreur inattendue.
    error_log("Erreur inattendue lors de la génération du BL (ID: " . $orderId . ") : " . $e->getMessage());
    die("Une erreur inattendue est survenue lors de la génération du bon de livraison.");
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    <title>Bon de Livraison N°<?php echo htmlspecialchars($formattedDeliveryId); ?></title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    <style>
        body {
            font-family: 'Arial', sans-serif;
            margin: 0;
            padding: 25px;
            background-color: #f8f8f8;
            color: #333;
            line-height: 1.6;
        }
        
        .document-container {
            width: 100%;
            max-width: 850px;
            margin: 20px auto;
            padding: 40px;
            background-color: #fff;
            border: 1px solid #e0e0e0;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.08);
            position: relative;
            box-sizing: border-box;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 3px solid #0056b3;
            padding-bottom: 20px;
            margin-bottom: 30px;
        }
        .header img {
            max-height: 90px;
        }
        .header h1 {
            color: #0056b3;
            font-size: 2.5em;
            margin: 0;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        .company-info, .order-summary {
            margin-bottom: 25px;
            font-size: 0.7em;
        }
        .company-info p, .order-summary p {
            margin: 7px 0;
        }
        .order-summary {
            display: flex;
            justify-content: space-between;
            margin-bottom: 30px;
            padding: 10px 0;
            border-top: 1px dashed #ccc;
            border-bottom: 1px dashed #ccc;
        }
        .order-summary div {
            width: 49%;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 30px;
            font-size: 0.7em;
            table-layout: fixed; /* Très important pour que les largeurs de colonne fonctionnent */
        }
        th, td {
            border: 1px solid #ddd;
            padding: 12px;
            text-align: left;
            word-wrap: break-word; /* Permet aux mots longs de se casser */
        }
        th {
            background-color: #007BFF;
            color: #fff;
            text-transform: uppercase;
            font-weight: 700;
            white-space: nowrap;
        }
        /* Ajustements de largeur des colonnes */
        th:nth-child(1), td:nth-child(1) { /* Nom Pièce */
            width: 25%;
        }
        th:nth-child(2), td:nth-child(2) { /* Référence */
            width: 15%;
        }
        th:nth-child(3), td:nth-child(3) { /* Qté Commandée */
            width: 10%;
        }
        th:nth-child(4), td:nth-child(4) { /* Qté Livrée */
            width: 10%;
        }
        th:nth-child(5), td:nth-child(5) { /* Prix U. HT */
            width: 13%;
        }
        th:nth-child(6), td:nth-child(6) { /* TVA (%) */
            width: 7%;
        }
        th:nth-child(7), td:nth-child(7) { /* Total TTC Livrée */
            width: 20%;
        }

        tbody tr:nth-child(even) {
            background-color: #f3f9ff;
        }
        tbody tr:hover {
            background-color: #e6f2ff;
        }
        .text-right { text-align: right; }
        .text-center { text-align: center; }

        .total-row th, .total-row td {
            background-color: #0056b3;
            color: #fff;
            font-weight: bold;
            font-size: 0.9em;
            padding: 15px 12px;
        }

        .action-buttons {
            text-align: center;
            margin-top: 30px;
            padding-top: 15px;
            border-top: 1px solid #eee;
        }
        .action-buttons button {
            padding: 12px 25px;
            font-size: 1em;
            cursor: pointer;
            background-color: #007BFF;
            color: white;
            border: none;
            border-radius: 6px;
            margin: 8px;
            transition: background-color 0.3s ease, transform 0.2s ease;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }
        .action-buttons button:hover {
            background-color: #0056b3;
            transform: translateY(-2px);
        }
        .action-buttons button:active {
            transform: translateY(0);
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.2);
        }

        .footer {
            text-align: center;
            margin-top: 50px;
            padding-top: 25px;
            border-top: 1px solid #e0e0e0;
            font-size: 0.8em;
            color: #777;
        }

        /* Styles d'impression */
        @media print {
            body {
                background-color: #fff;
                padding: 0;
            }
            .document-container {
                box-shadow: none;
                border: none;
                margin: 0;
                padding: 0;
                max-width: 100%;
                page-break-after: auto; /* Permet aux pages de se couper correctement */
            }
            .action-buttons {
                display: none !important; /* Cache les boutons à l'impression */
            }
            table {
                font-size: 0.75em;
                width: 100%;
                table-layout: fixed;
            }
            th, td {
                padding: 6px;
                word-wrap: break-word;
            }
            th:nth-child(1), td:nth-child(1) { width: 25%; }
            th:nth-child(2), td:nth-child(2) { width: 15%; }
            th:nth-child(3), td:nth-child(3) { width: 10%; }
            th:nth-child(4), td:nth-child(4) { width: 10%; }
            th:nth-child(5), td:nth-child(5) { width: 13%; }
            th:nth-child(6), td:nth-child(6) { width: 7%; }
            th:nth-child(7), td:nth-child(7) { width: 20%; }

            .header h1 {
                font-size: 1.8em;
            }
            .total-row th, .total-row td {
                font-size: 1em;
            }
        }
    </style>
</head>
<body>

<div class="document-container" id="pdfContent">
    <div class="header">
        <img src="img/logo_ase.png" alt="Logo ASE Maintenance">
        <h1>BON DE LIVRAISON</h1>
    </div>

    <div class="company-info">
        <h3>ASE Maintenance</h3>
        <p>Adresse : Toamasina, rue de commerce</p>
        <p>Téléphone : +261 34 05 216 90</p>
        <p>Email : Sales@ase.mg</p>
        <p>NIF/STAT: 3002015172 / 77100312017000183</p>
    </div>

    <div class="order-summary">
        <div>
            <p><strong>Bon de Livraison N°:</strong> <?php echo htmlspecialchars($formattedDeliveryId); ?></p>
            <p><strong>Date de livraison :</strong> <?php echo date('d/m/Y H:i:s'); ?> (Généré le)</p>
        </div>
        <div style="margin-left: 350px;">
            <p><strong>Numéro de commande d'origine :</strong> <?php echo htmlspecialchars($order['id_commande']); ?></p>
            <p><strong>Magasinier :</strong> <?php echo htmlspecialchars($order['magasinier_nom']); ?></p>
        </div>
    </div>

    <table>
        <thead>
            <tr>
                <th>Nom Pièce</th>
                <th>Référence</th>
                <th class="text-center">Qté Comm.</th>
                <th class="text-center">Qté Livr.</th>
                <th class="text-right">Prix U. HT</th>
                <th class="text-right">TVA (%)</th>
                <th class="text-right">Total TTC</th>
            </tr>
        </thead>
        <tbody>
            <?php
            $grandTotalTTCLivraison = 0;
            foreach ($orderDetails as $item) {
                $prixHT = $item['prix_ht_unitaire'];
                $tva = $item['tva_unitaire'];
                $quantiteLivree = $item['quantite_recue'];
                $totalTTCLigneLivree = ($prixHT * $quantiteLivree) * (1 + ($tva / 100));
                $grandTotalTTCLivraison += $totalTTCLigneLivree;
            ?>
                <tr>
                    <td><?php echo htmlspecialchars($item['nom_piece']); ?></td>
                    <td><?php echo htmlspecialchars($item['reference']); ?></td>
                    <td class="text-center"><?php echo htmlspecialchars($item['quantite_commandee']); ?></td>
                    <td class="text-center"><?php echo htmlspecialchars($item['quantite_recue']); ?></td>
                    <td class="text-right"><?php echo number_format($item['prix_ht_unitaire'], 2, ',', ' '); ?></td>
                    <td class="text-right"><?php echo number_format($item['tva_unitaire'], 0, ',', ' '); ?>%</td>
                    <td class="text-right"><?php echo number_format($totalTTCLigneLivree, 2, ',', ' '); ?></td>
                </tr>
            <?php } ?>
        </tbody>
        <tfoot>
            <tr class="total-row">
                <th colspan="6" class="text-right">TOTAL LIVRAISON TTC</th>
                <td class="text-right"><?php echo number_format($grandTotalTTCLivraison, 2, ',', ' '); ?> Ar</td>
            </tr>
        </tfoot>
    </table>

    <div class="footer">
        Généré le <?php echo date('d/m/Y H:i:s'); ?> par ASE Maintenance.
    </div>
</div>

<div class="action-buttons">
    <button onclick="downloadPdf()">Télécharger en PDF</button>
    <button onclick="window.close()">Fermer</button>
</div>

<script>
    function downloadPdf() {
        const { jsPDF } = window.jspdf;
        const doc = new jsPDF('p', 'mm', 'a4');

        const element = document.getElementById('pdfContent');

        // Masquer les boutons avant la capture pour qu'ils n'apparaissent pas dans le PDF
        const actionButtons = document.querySelector('.action-buttons');
        if (actionButtons) {
            actionButtons.style.display = 'none';
        }

        html2canvas(element, { scale: 2 }).then(canvas => {
            const imgData = canvas.toDataURL('image/png');
            const imgWidth = 210; // Largeur A4 en mm
            const pageHeight = 297; // Hauteur A4 en mm
            const imgHeight = canvas.height * imgWidth / canvas.width;
            let heightLeft = imgHeight;

            let position = 0;

            // Ajouter la première page
            doc.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
            heightLeft -= pageHeight;

            // Ajouter des pages supplémentaires si le contenu dépasse une page
            while (heightLeft > -10) { // Petite tolérance pour les fins de page
                position = heightLeft - imgHeight;
                doc.addPage();
                doc.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
                heightLeft -= pageHeight;
            }

            const filename = "Bon_de_Livraison_N" + "<?php echo $formattedDeliveryId; ?>" + ".pdf";
            doc.save(filename);

            // Rétablir l'affichage des boutons après le téléchargement ou l'erreur
            if (actionButtons) {
                actionButtons.style.display = 'block'; // Ou 'flex' selon votre CSS initial
            }
        }).catch(error => {
            console.error("Erreur lors de la capture du document avec html2canvas:", error);
            alert("Erreur lors de la génération du PDF. Veuillez réessayer.");
            // Assurez-vous que les boutons sont rétablis même en cas d'erreur
            if (actionButtons) {
                actionButtons.style.display = 'block';
            }
        });
    }
</script>

</body>
</html>