<?php
// ASE_Maintenance/generate_sorties_globales_excel.php

session_start();
require_once 'php/config.php';

// Définit le fuseau horaire
date_default_timezone_set('Indian/Antananarivo');

// Sécurité : Vérifier si l'utilisateur est connecté et a le bon rôle
if (
    !isset($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Administrateur'
) {
    header('HTTP/1.1 403 Forbidden');
    echo "Accès non autorisé.";
    exit();
}

// AJOUTÉ : Intégration de votre fonction de formatage pour les fiches d'intervention
function formatInterventionId($id_numeric) {
    if (!is_numeric($id_numeric) || $id_numeric <= 0) {
        return "INVALID_ID";
    }

    $base = 26; // Nombre de lettres dans l'alphabet (A-Z)
    $num_per_letter = 999; // Nombre de numéros par lettre (001 à 999)

    $letter_index_0_based = floor(($id_numeric - 1) / $num_per_letter);
    
    $letter_code = '';
    $temp_index = $letter_index_0_based;

    if ($temp_index < 0) return "INVALID_ID"; // Sécurité

    if ($temp_index == 0) {
        $letter_code = 'A';
    } else {
        while ($temp_index >= 0) {
            $letter_code = chr(65 + ($temp_index % $base)) . $letter_code;
            $temp_index = floor($temp_index / $base) - 1;
        }
    }

    $remainder = ($id_numeric - 1) % $num_per_letter;
    $numeric_part = str_pad($remainder + 1, 3, '0', STR_PAD_LEFT);

    return $letter_code . $numeric_part;
}


$pdo = getDbConnection();

// Fonction pour construire la clause WHERE (copiée pour être autonome)
function buildFilterClause($filter_vehicule_sigle, $filter_start_date, $filter_end_date, $filter_id_client) {
    $sql_where = " WHERE 1=1 ";
    $params = [];

    if (!empty($filter_vehicule_sigle) && $filter_vehicule_sigle !== 'all') {
        $sql_where .= " AND v.sigle = ?";
        $params[] = $filter_vehicule_sigle;
    }
    if (!empty($filter_id_client) && $filter_id_client !== 'all') {
        $sql_where .= " AND cli.id_client = ?";
        $params[] = $filter_id_client;
    }
    if (!empty($filter_start_date)) {
        $sql_where .= " AND sp.date_sortie >= ?";
        $params[] = $filter_start_date;
    }
    if (!empty($filter_end_date)) {
        $sql_where .= " AND sp.date_sortie <= ?";
        $params[] = $filter_end_date;
    }

    return ['sql_where' => $sql_where, 'params' => $params];
}

// Récupération des filtres depuis l'URL
$filter_vehicule_sigle = trim($_GET['filter_vehicule_sigle'] ?? '');
$filter_start_date = trim($_GET['filter_start_date'] ?? '');
$filter_end_date = trim($_GET['filter_end_date'] ?? '');
$filter_id_client = trim($_GET['filter_id_client'] ?? '');

$filter_data = buildFilterClause($filter_vehicule_sigle, $filter_start_date, $filter_end_date, $filter_id_client);
$sql_where = $filter_data['sql_where'];
$params = $filter_data['params'];

try {
    // Requête pour obtenir chaque ligne de détail de sortie de pièce
    $stmt = $pdo->prepare("
        SELECT
            sp.id_sortie,
            sp.date_sortie,
            i.id_intervention AS fiche_intervention_numero,
            i.date_intervention,
            v.sigle AS vehicule_sigle,
            v.immatriculation,
            cli.nom_client,
            p.nom_piece,
            dsp.quantite_sortie,
            dsp.prix_unitaire_ht_au_moment_sortie,
            (dsp.quantite_sortie * dsp.prix_unitaire_ht_au_moment_sortie) AS total_ht_piece
        FROM
            details_sortie_piece dsp
        JOIN sorties_pieces sp ON dsp.id_sortie = sp.id_sortie
        JOIN piece p ON dsp.id_piece = p.id_piece
        JOIN interventions i ON sp.id_intervention = i.id_intervention
        JOIN vehicules v ON i.id_vehicule = v.id_vehicule
        LEFT JOIN clients cli ON i.id_client = cli.id_client
        " . $sql_where . "
        ORDER BY sp.date_sortie ASC, sp.id_sortie ASC -- MODIFIÉ : Tri du plus ancien au plus récent
    ");

    $stmt->execute($params);
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Nom du fichier Excel
    $filename = "rapport_sorties_globales_" . date('Y-m-d') . ".xls";

    // Headers pour forcer le téléchargement
    header("Content-Type: application/vnd.ms-excel; charset=utf-8");
    header("Content-Disposition: attachment; filename=\"$filename\"");
    header("Pragma: no-cache");
    header("Expires: 0");

    $output = fopen('php://output', 'w');
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

    // MODIFIÉ : Entêtes du tableau Excel
    $headers = [
        'N° Fiche de Sortie',
        'Date Sortie',
        'N° Fiche Intervention',
        'Date Intervention',
        'Client',
        'Sigle Vehicule',
        'Immatriculation',
        'Piece',
        'Quantite Sortie',
        'Prix Unitaire HT (Ar)',
        'Total HT Ligne (Ar)'
    ];
    fputcsv($output, $headers, ';');

    if ($results) {
        foreach ($results as $row) {
            // MODIFIÉ : Application du formatage pour les numéros de fiches
            fputcsv($output, [
                'S' . sprintf('%03d', $row['id_sortie']), // Format S001, S002
                (new DateTime($row['date_sortie']))->format('d/m/Y H:i'),
                formatInterventionId($row['fiche_intervention_numero']), // Utilisation de votre fonction
                (new DateTime($row['date_intervention']))->format('d/m/Y'),
                $row['nom_client'] ?? 'N/A',
                $row['vehicule_sigle'] ?? 'N/A',
                $row['immatriculation'],
                $row['nom_piece'],
                $row['quantite_sortie'],
                number_format($row['prix_unitaire_ht_au_moment_sortie'], 2, ',', ' '),
                number_format($row['total_ht_piece'], 2, ',', ' ')
            ], ';');
        }
    }

    fclose($output);
    exit();

} catch (PDOException $e) {
    header('HTTP/1.1 500 Internal Server Error');
    die("Erreur de base de données lors de la génération du fichier Excel : " . $e->getMessage());
}
?>