<?php
// ASE_Maintenance/generate_rapport_pneumatique_pdf.php

session_start();

// Inclusion du fichier de configuration pour la connexion à la base de données
require_once 'php/config.php';

// Inclusion de la bibliothèque Dompdf
require_once 'dompdf-3.1.0/dompdf/vendor/autoload.php';
use Dompdf\Dompdf;
use Dompdf\Options;

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Administrateur ou Magasinier
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || ($_SESSION['role'] !== 'Administrateur' && $_SESSION['role'] !== 'Magasinier')) {
    header('Location: index.php'); // Rediriger si non autorisé
    exit();
}

$pdo = getDbConnection();

// Récupérer les paramètres de filtre depuis l'URL
$filter_model = trim($_GET['filter_model'] ?? 'all');
$filter_brand = trim($_GET['filter_brand'] ?? 'all');
$filter_dimension = trim($_GET['filter_dimension'] ?? 'all');
$filter_status = trim($_GET['filter_status'] ?? 'all');
$filter_vehicule_sigle = trim($_GET['filter_vehicule_sigle'] ?? 'all');
$filter_start_date = trim($_GET['filter_start_date'] ?? '');
$filter_end_date = trim($_GET['filter_end_date'] ?? '');

// Fonction pour construire les clauses WHERE et les paramètres pour les filtres
function buildFilterClausesForPneumaticReportPDF($filter_model, $filter_brand, $filter_dimension, $filter_status, $filter_vehicule_sigle, $filter_start_date, $filter_end_date) {
    $filters = [
        'pneu_model' => (!empty($filter_model) && $filter_model !== 'all') ? $filter_model : null,
        'pneu_brand' => (!empty($filter_brand) && $filter_brand !== 'all') ? $filter_brand : null,
        'pneu_dimension' => (!empty($filter_dimension) && $filter_dimension !== 'all') ? $filter_dimension : null,
        'pneu_status' => (!empty($filter_status) && $filter_status !== 'all') ? $filter_status : null,
        'vehicule_sigle' => (!empty($filter_vehicule_sigle) && $filter_vehicule_sigle !== 'all') ? $filter_vehicule_sigle : null,
        'start_date' => (!empty($filter_start_date)) ? $filter_start_date : null,
        'end_date' => (!empty($filter_end_date)) ? $filter_end_date : null,
    ];

    $generate_raw_conditions = function($alias_prefix, $date_col_name, $include_status = false, $include_vehicule_sigle = false) use ($filters) {
        $conditions_array = [];
        $params_array = [];

        if ($filters['pneu_model']) {
            $conditions_array[] = "{$alias_prefix}p.nom_modele = ?";
            $params_array[] = $filters['pneu_model'];
        }
        if ($filters['pneu_brand']) {
            $conditions_array[] = "{$alias_prefix}mp.nom_marque = ?";
            $params_array[] = $filters['pneu_brand'];
        }
        if ($filters['pneu_dimension']) {
            $conditions_array[] = "{$alias_prefix}dp.valeur_dimension = ?";
            $params_array[] = $filters['pneu_dimension'];
        }
        if ($include_status && $filters['pneu_status']) {
            $conditions_array[] = "{$alias_prefix}ip.statut_pneu = ?";
            $params_array[] = $filters['pneu_status'];
        }
        if ($include_vehicule_sigle && $filters['vehicule_sigle']) {
             $conditions_array[] = "{$alias_prefix}v.sigle = ?";
             $params_array[] = $filters['vehicule_sigle'];
        }
        if ($filters['start_date']) {
            $conditions_array[] = "{$date_col_name} >= ?";
            $params_array[] = $filters['start_date'];
        }
        if ($filters['end_date']) {
            $conditions_array[] = "{$date_col_name} <= ?";
            $params_array[] = $filters['end_date'];
        }

        return ['conditions_array' => $conditions_array, 'params' => $params_array];
    };

    return [
        'reception' => $generate_raw_conditions('', 'rp.date_reception'),
        'sortie_generale' => $generate_raw_conditions('', 'sp.date_sortie'),
        'montage_vehicule' => $generate_raw_conditions('', 'mpv.date_montage', true, true),
        'demontage_vehicule' => $generate_raw_conditions('', 'mpv.date_demontage', true, true), // Not used for PDF, but kept for consistency
        'current_stock' => $generate_raw_conditions('', 'ip.date_derniere_mise_a_jour', true, false)
    ];
}

$filter_clauses = buildFilterClausesForPneumaticReportPDF($filter_model, $filter_brand, $filter_dimension, $filter_status, $filter_vehicule_sigle, $filter_start_date, $filter_end_date);

// Seule la variable des pneus montés par véhicule est nécessaire
$mounted_tires_by_vehicle = [];
$grand_total_ttc = 0; // Variable pour le total général TTC

try {
    $append_where_and_params = function($base_sql, $conditions_array, $params_array) {
        $sql = $base_sql;
        if (!empty($conditions_array)) {
            if (strpos(strtoupper($base_sql), 'WHERE') === false) {
                $sql .= " WHERE " . implode(" AND ", $conditions_array);
            } else {
                $sql .= " AND " . implode(" AND ", $conditions_array);
            }
        }
        return ['sql' => $sql, 'params' => $params_array];
    };

    // Pneus montés par véhicule (avec calcul HT et TTC)
    $mounted_by_vehicle_query_parts = $filter_clauses['montage_vehicule'];
    $processed_mounted_by_vehicle_query = $append_where_and_params("
        SELECT
            v.immatriculation,
            v.sigle,
            SUM(p.prix_achat_ht) AS total_mounted_ht,
            SUM(p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS total_mounted_ttc,
            SUM(p.prix_achat_ht * (p.tva_applicable / 100)) AS total_mounted_tva,
            COUNT(mpv.id_montage) AS total_mounted_count
        FROM
            montage_pneu_vehicule mpv
        JOIN inventaire_pneu ip ON mpv.id_inventaire_pneu = ip.id_inventaire_pneu
        JOIN pneus p ON ip.id_pneu = p.id_pneu
        JOIN marques_pneu mp ON p.id_marque = mp.id_marque
        JOIN dimensions_pneu dp ON p.id_dimension = dp.id_dimension
        JOIN vehicules v ON mpv.id_vehicule = v.id_vehicule
        WHERE mpv.date_demontage IS NULL
    ", $mounted_by_vehicle_query_parts['conditions_array'], $mounted_by_vehicle_query_parts['params']);

    $stmt_mounted_by_vehicle = $pdo->prepare($processed_mounted_by_vehicle_query['sql'] . " GROUP BY v.id_vehicule, v.immatriculation, v.sigle ORDER BY v.immatriculation ASC");
    $stmt_mounted_by_vehicle->execute($processed_mounted_by_vehicle_query['params']);
    $mounted_tires_by_vehicle = $stmt_mounted_by_vehicle->fetchAll(PDO::FETCH_ASSOC);

    // Calcul du total général TTC
    foreach ($mounted_tires_by_vehicle as $item) {
        $grand_total_ttc += $item['total_mounted_ttc'];
    }

} catch (PDOException $e) {
    error_log("Erreur PDO lors de la génération du PDF du rapport pneumatique: " . $e->getMessage());
    die("Erreur de base de données lors de la génération du PDF: " . $e->getMessage());
}

// Fonction de conversion de nombre en toutes lettres (simplifiée pour les besoins du rapport)
function numberToFrenchWords($number) {
    $units = array('', 'un', 'deux', 'trois', 'quatre', 'cinq', 'six', 'sept', 'huit', 'neuf');
    $tens = array('', 'dix', 'vingt', 'trente', 'quarante', 'cinquante', 'soixante', 'soixante-dix', 'quatre-vingt', 'quatre-vingt-dix');
    $teens = array('dix', 'onze', 'douze', 'treize', 'quatorze', 'quinze', 'seize');

    $words = array();
    $number = (string) $number;

    if ($number == 0) return 'zéro';

    $parts = explode('.', $number);
    $integer_part = (int)$parts[0];
    $decimal_part = isset($parts[1]) ? (int)$parts[1] : 0;

    $integer_words = [];
    $millions = floor($integer_part / 1000000);
    $thousands = floor(($integer_part % 1000000) / 1000);
    $hundreds = $integer_part % 1000;

    if ($millions > 0) {
        if ($millions == 1) {
            $integer_words[] = 'un million';
        } else {
            $integer_words[] = numberToFrenchWords($millions) . ' millions';
        }
    }

    if ($thousands > 0) {
        if ($thousands == 1) {
            $integer_words[] = 'mille';
        } else {
            $integer_words[] = numberToFrenchWords($thousands) . ' mille';
        }
    }

    if ($hundreds > 0) {
        $h_words = [];
        $h = floor($hundreds / 100);
        $r = $hundreds % 100;

        if ($h == 1) {
            $h_words[] = 'cent';
        } elseif ($h > 1) {
            $h_words[] = $units[$h] . ' cents';
            if ($r == 0) $h_words = [$units[$h] . ' cent'];
        }

        if ($r > 0) {
            if ($r < 17) {
                if ($r < 10) $h_words[] = $units[$r];
                else $h_words[] = $teens[$r - 10];
            } else {
                $t = floor($r / 10);
                $u = $r % 10;
                $h_words[] = $tens[$t];
                if ($u > 0) {
                    if ($t == 7 || $t == 9) {
                        $h_words[count($h_words)-1] .= '-' . $teens[$u];
                    } else if ($u == 1 && $t != 8) {
                        $h_words[count($h_words)-1] .= '-et-' . $units[$u];
                    } else {
                        $h_words[count($h_words)-1] .= '-' . $units[$u];
                    }
                }
            }
        }
        $integer_words[] = implode(' ', $h_words);
    }

    $final_integer_words = implode(' ', $integer_words);

    if ($decimal_part > 0) {
        return trim($final_integer_words) . ' et ' . numberToFrenchWords($decimal_part) . ' centimes';
    }

    return trim($final_integer_words);
}

// Préparer le logo en Base64 pour Dompdf
$logo_path = 'img/logo_ase.png';
$logo_base64 = '';
if (file_exists($logo_path)) {
    $type = pathinfo($logo_path, PATHINFO_EXTENSION);
    $data = file_get_contents($logo_path);
    $logo_base64 = 'data:image/' . $type . ';base64,' . base64_encode($data);
}

// Créer une instance de Dompdf
$options = new Options();
$options->set('isHtml5ParserEnabled', true);
$options->set('isRemoteEnabled', true);
$dompdf = new Dompdf($options);

ob_start();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Rapport Pneumatique</title>
    <style>
        html { height: 100%; }
        body {
            font-family: sans-serif;
            margin: 0;
            padding: 0;
            font-size: 10px;
            padding-bottom: 10mm;
        }

        @page { margin: 1cm; }

        .logo-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 70mm;
            height: 25mm;
            z-index: 1;
        }
        .logo-container img {
            max-width: 90%;
            height: auto;
            display: block;
        }

        .first-page-header-box {
            position: absolute;
            top: 0;
            right: 0;
            height: 20mm;
            width: 50mm;
            text-align: center;
            border: 1px solid black;
            padding: 8px 12px;
            font-size: 1.2em;
            font-weight: bold;
            box-sizing: border-box;
            margin-top: 10px;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 2;
        }
        .first-page-header-box p {
            margin: 2px 0;
            line-height: 1.4;
        }
        .first-page-header-box p:first-child {
            font-size: 1.3em;
            margin-bottom: 5px;
        }
        .document-container {
            width: 100%;
            position: relative;
            margin-top: 35mm;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 2mm;
            margin-bottom: 5mm;
            table-layout: fixed;
        }
        th, td {
            border: 1px solid black;
            padding: 8px 6px;
            text-align: left;
            font-size: 0.9em;
            vertical-align: top;
            word-wrap: break-word;
        }
        th {
            background-color: #e0e0e0;
            color: black;
            text-transform: uppercase;
            font-weight: bold;
            text-align: center;
        }
        thead { display: table-header-group; }

        .report-summary-box {
            margin-bottom: 10mm;
        }
        .report-summary-box strong {
            display: block;
            margin-bottom: 2mm;
        }
        .report-summary-box p {
            margin: 2px 0;
        }
        .text-info {
            width: 100%;
            display: table;
            margin-bottom: 5mm;
        }
        .text-info div {
            display: table-cell;
            width: 50%;
            vertical-align: top;
        }
        .no-data {
            text-align: center;
            margin-top: 20px;
            font-size: 1.2em;
        }

        .totals-summary-box {
            width: 40%;
            margin-left: auto;
            border: 1px solid black;
            padding: 5px;
            margin-top: 0mm;
            margin-bottom: 0mm;
            box-sizing: border-box;
            clear: both;
            page-break-inside: avoid;
            text-align: right;
        }
        .total-line {
            display: flex;
            justify-content: space-between;
            margin-bottom: 2px;
            padding: 2px 0;
        }
        .total-line .total-label {
            font-weight: bold;
            text-align: left;
            font-size: 1.0em;
        }
        .total-line .total-value {
            text-align: right;
            font-size: 1.1em;
        }
        .final-total .total-label,
        .final-total .total-value {
            font-size: 1.2em;
            font-weight: bold;
            padding-top: 0px;
            margin-top: 0px;
        }

        #footer-content {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            width: 100%;
            text-align: center;
            font-size: 0.7em;
            color: black;
            border-top: 1px solid black;
            padding-top: 5px;
            box-sizing: border-box;
            line-height: 1.4;
        }
        #footer-content p {
            margin: 2px 0;
        }
    </style>
</head>
<body>
    <div class="logo-container">
        <?php if (!empty($logo_base64)): ?>
            <img src="<?= $logo_base64 ?>" alt="Logo ASE">
        <?php else: ?>
            <p style="color: red; font-size: 0.8em;">Logo non trouvé</p>
        <?php endif; ?>
    </div>

    <div class="first-page-header-box">
        <p>RAPPORT PNEUMATIQUE</p>
        <p>Généré le <?= date('d/m/Y') ?></p>
    </div>

    <div class="document-container">
        <div class="text-info">
            <div>
                <p><strong>Généré par :</strong> <?php echo htmlspecialchars($_SESSION['username'] ?? 'N/A'); ?> (<?php echo htmlspecialchars($_SESSION['role'] ?? 'N/A'); ?>)</p>
                <p><strong>Date de génération :</strong> <?php echo date('d/m/Y H:i:s'); ?></p>
            </div>
            
        </div>
        <?php if (empty($mounted_tires_by_vehicle)): ?>
            <p class="no-data">Aucun pneu monté sur véhicule trouvé avec les filtres appliqués.</p>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Véhicule (Immatriculation / Sigle)</th>
                        <th>Nombre de Pneus Montés</th>
                        <th>Valeur Totale (HT)</th>
                        <th>Valeur Totale (TVA)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($mounted_tires_by_vehicle as $item): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($item['immatriculation'] . ' (' . ($item['sigle'] ?? 'N/A') . ')'); ?></td>
                            <td style="text-align: center;"><?php echo htmlspecialchars($item['total_mounted_count']); ?></td>
                            <td style="text-align: center;"><?php echo number_format($item['total_mounted_ht'], 2, ',', '.') ; ?> Ar</td>
                            <td style="text-align: center;"><?php echo number_format($item['total_mounted_tva'], 2, ',', '.') ; ?> Ar</td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <div class="totals-summary-box">
                <div class="total-line final-total">
                    <span class="total-label"><strong>TOTAL TTC:</strong></span>
                    <span class="total-value"><strong><?php echo number_format($grand_total_ttc, 2, ',', '.') . ' Ar'; ?></strong></span>
                </div>
            </div>
            <p style="margin-top: 10px; font-size: 0.9em; font-style: italic;">Arrêté le présent rapport à la somme de : <?php echo utf8_decode(ucfirst(numberToFrenchWords($grand_total_ttc))); ?> Ariary TTC.</p>
        <?php endif; ?>
    </div>

    <div id="footer-content">
        <p><strong>AUTO SERVICE DE L'EST SARL</strong></p>
        <p>NIF: 3002015172 - STAT: 77100312000183</p>
        <p>Rue de commerce Toamasina 501 - Tél: +261 34 05 216 90 - Email: contact@ase.mg</p>
        <p>RIB BGFI: 00012 01250 41015401011 63 - RIB BFV: 00008 00490 02001012003 70</p>
    </div>
</body>
</html>
<?php
$html = ob_get_clean();
$dompdf->loadHtml($html);
$dompdf->setPaper('A4', 'portrait');
$dompdf->render();

$filename = "Rapport_Pneumatique_" . date('Ymd_His') . ".pdf";
$dompdf->stream($filename, ["Attachment" => true]);
exit();
?>