<?php
// generate_pdf_sortie_quantite_seulement.php
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Pragma: no-cache");
header("Expires: 0");
// Inclure le fichier de configuration de la base de données
require_once __DIR__ . '/php/config.php';

// Fonction pour obtenir la connexion à la base de données
$pdo = getDbConnection();

// Fonction pour formater l'ID d'intervention en A001, B001, etc.
// Copiée depuis generate_pdf_intervention.php pour cohérence
function formatInterventionId($id_numeric) {
    if (!is_numeric($id_numeric) || $id_numeric <= 0) {
        return "INVALID_ID";
    }

    $base = 26; // Nombre de lettres dans l'alphabet (A-Z)
    $num_per_letter = 999; // Nombre de numéros par lettre (001 à 999)

    // Calculer l'indice de la lettre (0-basé: A=0, B=1, ..., Z=25, AA=26, ...)
    $letter_index_0_based = floor(($id_numeric - 1) / $num_per_letter);
    
    $letter_code = '';
    $temp_index = $letter_index_0_based;

    do {
        $char_value = ($temp_index % $base);
        $letter_code = chr(65 + $char_value) . $letter_code;
        $temp_index = floor($temp_index / $base);
        if ($char_value == ($base - 1) && $temp_index > 0) { 
             $temp_index--;
        }
    } while ($temp_index > 0); 
    
    if (empty($letter_code) && $id_numeric <= $num_per_letter) {
        $letter_code = 'A';
    }


    // Formater la partie numérique (001 à 999)
    $remainder = ($id_numeric - 1) % $num_per_letter;
    $numeric_part = str_pad($remainder + 1, 3, '0', STR_PAD_LEFT);

    return $letter_code . $numeric_part;
}


// Vérifier si l'ID de la sortie est passé en paramètre GET
if (!isset($_GET['id_sortie']) || !filter_var($_GET['id_sortie'], FILTER_VALIDATE_INT)) {
    die('ID de sortie manquant ou invalide.');
}

$id_sortie_numeric = filter_var($_GET['id_sortie'], FILTER_VALIDATE_INT);

// Récupérer les données de la sortie, l'intervention associée, le véhicule, le client et le magasinier
try {
    $stmt = $pdo->prepare("
        SELECT
            sp.id_sortie,
            sp.date_sortie,
            sp.notes AS notes_sortie,
            sp.total_ht,
            sp.total_tva,
            sp.total_ttc,
            u.nom_utilisateur AS magasinier_nom,
            i.id_intervention,
            i.date_intervention,
            i.kilometrage_intervention,
            i.origine_panne,
            i.rapport_intervention,
            i.statut,
            v.immatriculation,
            v.sigle,
            v.marque,
            v.vin,
            v.kilometrage AS vehicule_kilometrage,
            c.nom_client,
            c.contact_client,
            c.email_client,
            ch.nom_chauffeur
        FROM
            sorties_pieces sp
        JOIN
            utilisateurs u ON sp.id_magasinier = u.id_utilisateur
        JOIN
            interventions i ON sp.id_intervention = i.id_intervention
        JOIN
            vehicules v ON i.id_vehicule = v.id_vehicule
        JOIN
            clients c ON v.id_client = c.id_client
        LEFT JOIN
            chauffeurs ch ON v.id_vehicule = ch.id_vehicule
        WHERE
            sp.id_sortie = ?
    ");
    $stmt->execute([$id_sortie_numeric]);
    $sortie = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$sortie) {
        die('Sortie de pièces introuvable.');
    }

    // --- Récupérer les détails des pièces sorties pour cette sortie ---
    $stmt_pieces = $pdo->prepare("
        SELECT
            dsp.quantite_sortie,
            dsp.prix_unitaire_ht_au_moment_sortie,
            dsp.tva_taux_au_moment_sortie,
            p.nom_piece,
            p.reference,
            p.code_piece
        FROM
            details_sortie_piece dsp
        JOIN
            piece p ON dsp.id_piece = p.id_piece
        WHERE
            dsp.id_sortie = ?
        ORDER BY p.nom_piece ASC
    ");
    $stmt_pieces->execute([$id_sortie_numeric]);
    $pieces_sorties = $stmt_pieces->fetchAll(PDO::FETCH_ASSOC);

    $nombre_articles_total = 0;
    foreach ($pieces_sorties as $piece) {
        $nombre_articles_total += $piece['quantite_sortie'];
    }

} catch (PDOException $e) {
    error_log("Erreur PDO lors de la récupération des données pour la fiche de sortie : " . $e->getMessage());
    die('Erreur de base de données lors de la récupération des données.');
}

// Préparer les données pour l'affichage HTML
$date_sortie = (new DateTime($sortie['date_sortie']))->format('d/m/y H:i');
$num_sortie = htmlspecialchars('S' . sprintf('%03d', $sortie['id_sortie'])); // Format S001, S002
$nom_magasinier = htmlspecialchars($sortie['magasinier_nom'] ?? 'N/A');
// Décoder les entités HTML avant d'appliquer htmlspecialchars pour l'affichage
$notes_sortie = htmlspecialchars(html_entity_decode($sortie['notes_sortie'] ?? 'Aucune note.', ENT_QUOTES | ENT_HTML5, 'UTF-8'));


// Détails de l'intervention liée
// Utilisation de la fonction formatInterventionId pour le numéro d'intervention
$num_intervention_formatted = formatInterventionId($sortie['id_intervention']);
$num_intervention = htmlspecialchars(' ' . $num_intervention_formatted); 

$demandeur_intervention = htmlspecialchars($sortie['nom_client']);
$num_vehicule_intervention = htmlspecialchars($sortie['immatriculation']);
$sig_vehicule_intervention = htmlspecialchars($sortie['sigle']);
$marque_vehicule_intervention = htmlspecialchars($sortie['marque']);
$kilometrage_intervention = htmlspecialchars($sortie['kilometrage_intervention']) . ' Km';
$origine_panne_intervention = htmlspecialchars($sortie['origine_panne']);
$rapport_intervention = htmlspecialchars($sortie['rapport_intervention'] ?? 'Aucun rapport détaillé.');
$nom_chauffeur = htmlspecialchars($sortie['nom_chauffeur'] ?? 'N/A');

$date_creation_fiche_intervention = htmlspecialchars((new DateTime($sortie['date_intervention']))->format('d/m/y')); // Date de l'intervention liée
$date_fiche_sortie = htmlspecialchars((new DateTime($sortie['date_sortie']))->format('d/m/y')); // Date de la fiche de sortie

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    <title>Fiche de Sortie N°<?php echo $num_sortie; ?> (Quantités)</title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 0;
            background-color: #f4f4f4;
            display: flex;
            justify-content: center;
            align-items: flex-start;
            min-height: 100vh;
        }
        .fiche-container {
            width: 210mm; /* Largeur A4 */
            height: 297mm; /* Hauteur A4 Fixe pour forcer une seule page */
            overflow: hidden;
            background-color: #fff;
            border: 1px solid #ccc;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
            padding: 15mm;
            box-sizing: border-box;
            font-size: 9.5pt;
            color: #333;
        }
        .header {
            text-align: center;
            margin-bottom: 8mm;
        }
        .header h1 {
            font-size: 16pt;
            margin: 0;
            padding: 0;
            color: #000;
        }
        .info-grid {
            display: grid;
            grid-template-columns: repeat(6, 1fr);
            gap: 2px 5mm;
            margin-bottom: 3mm;
            align-items: end;
        }
        .info-item {
            display: flex;
            flex-direction: column;
            gap: 1px;
        }
        .info-item label {
            font-weight: bold;
            font-size: 7pt;
            color: #555;
        }
        .info-item span {
            border-bottom: 1px solid #000;
            padding-bottom: 1px;
            min-height: 10px;
            box-sizing: border-box;
        }
        .info-section-title {
            background-color: #e0e0e0;
            font-weight: bold;
            text-align: center;
            padding: 2px;
            border: 1px solid #ccc;
            margin-top: 3mm;
            margin-bottom: 2mm;
            font-size: 9.5pt;
        }
        .content-box {
            border: 1px solid #000;
            min-height: 15mm;
            padding: 3px;
            box-sizing: border-box;
            margin-bottom: 3mm;
        }
        .text-area {
            min-height: 15mm;
            border: none;
            width: 100%;
            padding: 0;
            margin: 0;
            resize: none;
            overflow: hidden;
            background-color: transparent;
        }
        .signature-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10mm;
            margin-top: 5mm;
        }
        .signature-item {
            display: flex;
            flex-direction: column;
            align-items: center;
        }
        .signature-block {
            text-align: center;
            padding-top: 3mm;
            border-top: 1px solid #000;
            min-height: 10mm;
            width: 80%;
        }
        .signature-title {
            font-weight: bold;
            margin-bottom: 3px;
            font-size: 8.5pt;
        }
        .date-section {
            text-align: right;
            margin-top: 5mm;
            font-size: 8.5pt;
        }
        .date-section span {
            border-bottom: 1px solid #000;
            padding: 0 4mm 1px 4mm;
            display: inline-block;
        }

        /* Style pour le tableau de pièces */
        .parts-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1mm;
            margin-bottom: 3mm;
        }
        .parts-table th, .parts-table td {
            border: 1px solid #000;
            padding: 1.5px;
            text-align: left;
            font-size: 7.5pt;
        }
        .parts-table th {
            background-color: #e0e0e0;
            text-align: center;
        }
        .col-span-full {
            grid-column: span 6;
        }
        .col-span-3 {
            grid-column: span 3;
        }
        .col-span-2 {
            grid-column: span 2;
        }
        .col-span-4 {
            grid-column: span 4;
        }

        /* Styles pour l'indicateur de chargement */
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            display: flex;
            justify-content: center;
            align-items: center;
            flex-direction: column;
            color: white;
            font-size: 1.5em;
            z-index: 9999;
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s, visibility 0.3s;
        }
        .loading-overlay.visible {
            opacity: 1;
            visibility: visible;
        }
        .spinner {
            border: 4px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top: 4px solid #fff;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin-bottom: 15px;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
         /* Styles des boutons d'action - toujours visibles sur la page HTML mais ignorés par html2canvas */
        .action-buttons {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            display: flex;
            gap: 15px;
            z-index: 1000;
        }
        .action-buttons button {
            padding: 10px 20px;
            font-size: 16px;
            cursor: pointer;
            background-color: #007BFF;
            color: white;
            border: none;
            border-radius: 5px;
            transition: background-color 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        .action-buttons button:hover {
            background-color: #0056b3;
        }
        .action-buttons button:nth-child(2) {
            background-color: #28a745;
        }
        .action-buttons button:nth-child(2):hover {
            background-color: #218838;
        }
        .action-buttons button:nth-child(3) {
            background-color: #6c757d;
        }
        .action-buttons button:nth-child(3):hover {
            background-color: #5a6268;
        }
    </style>
</head>
<body>
    <div class="fiche-container" id="fiche-sortie">
        <div class="header">
            <!-- Vous pouvez ajouter un logo ici si nécessaire -->
            <!-- <img src="img/logo_ase.png" alt="Logo" style="height: 50px; margin-bottom: 10px;"> -->
            <h1>FICHE DE SORTIE DE PIÈCES</h1>
        </div>

        <!-- Section EN-TÊTE -->
        <div class="info-grid">
            <div class="info-item col-span-2">
                <label>Date Sortie</label>
                <span><?php echo $date_sortie; ?></span>
            </div>
            <div class="info-item col-span-1">
                <label>Date début intervention</label>
                <span><?php echo htmlspecialchars((new DateTime($sortie['date_intervention']))->format('d/m/y')); ?></span>
            </div>
            <div class="info-item col-span-1">
                <label>Statut Intervention</label>
                <span><?php echo htmlspecialchars($sortie['statut']); ?></span>
            </div>
            <div class="info-item col-span-1">
                <label>N° Intervention</label>
                <span><?php echo $num_intervention; ?></span>
            </div>
            <div class="info-item col-span-2">
                <label>N° Sortie</label>
                <span><?php echo $num_sortie; ?></span>
            </div>
            <div class="info-item col-span-2">
                <label>Magasinier</label>
                <span><?php echo $nom_magasinier; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>CLIENT (de l'intervention)</label>
                <span><?php echo $demandeur_intervention; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>Email Client</label>
                <span><?php echo htmlspecialchars($sortie['email_client'] ?? 'N/A'); ?></span>
            </div>

            <div class="info-item col-span-3">
                <label>N° VEHICULE (de l'intervention)</label>
                <span><?php echo $num_vehicule_intervention; ?>---<?php echo $sig_vehicule_intervention; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>Kilométrage Intervention</label>
                <span><?php echo $kilometrage_intervention; ?></span>
            </div>

            <div class="info-item col-span-3">
                <label>MARQUE VEHICULE</label>
                <span><?php echo $marque_vehicule_intervention; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>VIN Véhicule</label>
                <span><?php echo htmlspecialchars($sortie['vin'] ?? 'N/A'); ?></span>
            </div>

            <div class="info-item col-span-3">
                <label>CONDUCTEUR (véhicule)</label>
                <span><?php echo $nom_chauffeur; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>Contact Client</label>
                <span><?php echo htmlspecialchars($sortie['contact_client'] ?? 'N/A'); ?></span>
            </div>
        </div>

        <p style="margin-bottom: 3mm; font-size: 7pt;">
            Cette fiche détaille les pièces sorties du stock pour l'intervention référencée ci-dessus. Elle sert de document de suivi pour la consommation des pièces. Les informations de prix ne sont pas affichées sur ce document.
        </p>

        <!-- Section NATURE DE LA DEMANDE (issue de l'intervention liée) -->
        <div class="info-section-title">NATURE DE LA DEMANDE (Intervention liée)</div>
        <div class="info-grid" style="grid-template-columns: 1fr;">
            <div class="info-item col-span-full">
                <label>Origine de la Panne / Motif de l'Intervention</label>
                <span class="content-box" style="min-height: 20mm;"><?php echo nl2br($origine_panne_intervention); ?></span>
            </div>
        </div>

        <!-- Section DÉTAILS DE L'INTERVENTION (rapport de l'intervention liée) -->
        <div class="info-section-title">RAPPORT D'INTERVENTION (Intervention liée)</div>
        <div class="info-grid" style="grid-template-columns: 1fr;">
            <div class="info-item col-span-full">
                <label>Description des Travaux Effectués</label>
                <span class="content-box" style="min-height: 20mm;"><?php echo nl2br($rapport_intervention); ?></span>
            </div>
        </div>

        <!-- Section OBSERVATIONS (notes spécifiques à la sortie) -->
        <div class="info-section-title">NOTES SPÉCIFIQUES À LA SORTIE</div>
        <div class="content-box" style="min-height: 15mm;"><?php echo nl2br($notes_sortie); ?></div>

        <!-- Section PRODUITS/PIECES CHANGEES -->
        <div class="info-section-title">PRODUITS/PIECES SORTIES</div>
        <table class="parts-table">
            <thead>
                <tr>
                    <th style="width: 5%;">N°</th>
                    <th style="width: 30%;">Désignation</th>
                    <th style="width: 25%;">Référence</th>
                    <th style="width: 20%;">Code Pièce / Série</th>
                    <th style="width: 20%;">Quantité</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($pieces_sorties)): ?>
                    <tr>
                        <td colspan="5" style="text-align: center;">Aucune pièce enregistrée pour cette sortie.</td>
                    </tr>
                <?php else: ?>
                    <?php $i = 1; ?>
                    <?php foreach ($pieces_sorties as $piece): ?>
                        <tr>
                            <td><?php echo $i++; ?></td>
                            <td><?php echo htmlspecialchars($piece['nom_piece']); ?></td>
                            <td><?php echo htmlspecialchars($piece['reference']); ?></td>
                            <td><?php echo htmlspecialchars($piece['code_piece'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($piece['quantite_sortie']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
        <div style="text-align: right; margin-bottom: 5mm; font-size: 9pt;">
            <b>Nombre d'articles uniques :</b> <span style="border-bottom: 1px solid #000; padding: 0 10mm;"><?php echo count($pieces_sorties); ?></span>
            <br>
            <b>Quantité totale de pièces sorties :</b> <span style="border-bottom: 1px solid #000; padding: 0 10mm;"><?php echo $nombre_articles_total; ?></span><br><br>
            <!-- Ligne "Total TTC Sortie" masquée -->
            <!-- <b>Total TTC Sortie</b> <span  style="border-bottom: 1px solid #000; padding: 0 10mm;"><?php echo number_format($sortie['total_ttc'], 2, ',', '.') . ' Ar'; ?></span> -->
               
        </div>


        <!-- Section Signatures -->
        <div class="signature-grid">
            <div class="signature-item">
                <div class="signature-title">Nom et Visa du Magasinier</div>
                <div class="signature-block"><?php echo $nom_magasinier; ?></div>
            </div>
            <div class="signature-item">
                <div class="signature-title">Nom et Visa de Conducteur ou Client</div>
                <div class="signature-block"><?php echo $demandeur_intervention; ?></div>
            </div>
        </div>

        <div class="date-section">
            Fait à : <span></span> Le : <span><?php echo $date_fiche_sortie; ?></span>
        </div>
    </div>

    <div class="action-buttons">
        <button onclick="downloadPdf()"><i class="fas fa-file-pdf"></i> Télécharger en PDF</button>
        <button onclick="window.close()"><i class="fas fa-times-circle"></i> Fermer</button>
    </div>

    <!-- Loading Overlay -->
    <div id="loadingOverlay" class="loading-overlay">
        <div class="spinner"></div>
        <p>Génération du PDF en cours...</p>
    </div>

    <script>
        window.jsPDF = window.jspdf.jsPDF; // Correctement initialiser jsPDF

        function downloadPdf() {
            const { jsPDF } = window.jspdf;
            // Utiliser 'p' pour portrait, 'mm' pour millimètres, 'a4' pour le format
            const doc = new jsPDF('p', 'mm', 'a4');

            const element = document.getElementById('fiche-sortie'); // ID de la fiche de sortie
            const loadingOverlay = document.getElementById('loadingOverlay');

            // Cacher les boutons et afficher l'overlay de chargement
            const actionButtons = document.querySelector('.action-buttons');
            if (actionButtons) {
                actionButtons.style.display = 'none';
            }
            loadingOverlay.classList.add('visible');

            const options = {
                scale: 1.5, // Augmente l'échelle pour une meilleure qualité
                useCORS: true, // Important pour charger des images si elles viennent d'une autre origine
                logging: false, // Désactive les logs de html2canvas dans la console
                letterRendering: true, // Tente d'améliorer le rendu du texte
                // Ignorer les éléments des boutons d'action lors de la capture
                ignoreElements: (element) => {
                    return element.classList.contains('action-buttons');
                }
            };

            html2canvas(element, options).then(canvas => {
                const imgData = canvas.toDataURL('image/png');
                const imgWidth = 210; // Largeur A4 en mm
                const pageHeight = 297; // Hauteur A4 en mm
                const imgHeight = canvas.height * imgWidth / canvas.width;
                let heightLeft = imgHeight;

                let position = 0;

                doc.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
                heightLeft -= pageHeight;

                // Gérer les pages multiples si le contenu dépasse une page A4
                while (heightLeft > -10) { // Une petite marge pour éviter les coupures disgracieuses
                    position = heightLeft - imgHeight;
                    doc.addPage();
                    doc.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
                    heightLeft -= pageHeight;
                }

                // Utiliser l'ID de la sortie dans le nom du fichier PDF
                const sortieId = "<?php echo htmlspecialchars($id_sortie_numeric); ?>";
                // Nom de fichier adapté pour la version "quantité seulement"
                const filename = `fiche_sortie_pieces_quantites_S${String(sortieId).padStart(3, '0')}.pdf`;
                doc.save(filename);

            }).catch(error => {
                console.error("Erreur lors de la capture du document avec html2canvas:", error);
                alert("Une erreur est survenue lors de la génération du PDF. Veuillez réessayer.");
            }).finally(() => {
                // Toujours masquer l'overlay et réafficher les boutons, même en cas d'erreur
                loadingOverlay.classList.remove('visible');
                if (actionButtons) {
                    actionButtons.style.display = 'flex';
                }
            });
        }
    </script>
</body>
</html>
