<?php
// generate_pdf_intervention.php
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Pragma: no-cache");
header("Expires: 0");
// Inclure le fichier de configuration de la base de données
// Assurez-vous que le chemin vers 'php/config.php' est correct par rapport à ce fichier.
// Par exemple, si generate_pdf_intervention.php est à la racine et php/config.php est dans un sous-dossier:
require_once __DIR__ . '/php/config.php';

// Fonction pour obtenir la connexion à la base de données
$pdo = getDbConnection();

// Fonction pour formater l'ID d'intervention en A001, B001, etc.
function formatInterventionId($id_numeric) {
    if (!is_numeric($id_numeric) || $id_numeric <= 0) {
        return "INVALID_ID";
    }

    $base = 26; // Nombre de lettres dans l'alphabet (A-Z)
    $num_per_letter = 999; // Nombre de numéros par lettre (001 à 999)

    // Calculer l'indice de la lettre (0-basé: A=0, B=1, ..., Z=25, AA=26, ...)
    // L'ID numérique est ajusté de -1 car les indices de lettres commencent à 0 et les numéros à 1.
    $letter_index_0_based = floor(($id_numeric - 1) / $num_per_letter);
    
    $letter_code = '';
    $temp_index = $letter_index_0_based;

    // Convertir l'indice numérique en séquence de lettres (base 26)
    // Cette boucle gère les cas comme A, B, ..., Z, AA, AB, etc.
    do {
        $char_value = ($temp_index % $base);
        $letter_code = chr(65 + $char_value) . $letter_code; // Ajoute la lettre en préfixe
        $temp_index = floor($temp_index / $base);

        if ($char_value == ($base - 1) && $temp_index > 0) { 
             $temp_index--;
        }

    } while ($temp_index >= 0 && ($id_numeric - 1) >= $num_per_letter * $base); 
    if (empty($letter_code) || $id_numeric <= $num_per_letter) {
        $letter_code = 'A';
    }


    // Formater la partie numérique (001 à 999)
    $remainder = ($id_numeric - 1) % $num_per_letter;
    $numeric_part = str_pad($remainder + 1, 3, '0', STR_PAD_LEFT);

    return $letter_code . $numeric_part;
}


// Vérifier si l'ID de l'intervention est passé en paramètre GET
if (!isset($_GET['id_intervention']) || !filter_var($_GET['id_intervention'], FILTER_VALIDATE_INT)) {
    die('ID d\'intervention manquant ou invalide.');
}

$id_intervention_numeric = filter_var($_GET['id_intervention'], FILTER_VALIDATE_INT);
$id_intervention_formatted = formatInterventionId($id_intervention_numeric); // Formater l'ID

// Récupérer les données de l'intervention, du véhicule et du client
try {
    $stmt = $pdo->prepare("
        SELECT
            i.id_intervention,
            i.date_intervention,
            i.kilometrage_intervention,
            i.origine_panne,
            i.rapport_intervention,
            i.statut,
            i.date_creation,
            v.immatriculation,
            v.sigle,
            v.marque,
            v.vin,
            v.kilometrage AS vehicule_kilometrage,
            c.nom_client,
            c.contact_client,
            c.email_client,
            ch.nom_chauffeur -- Ajout du nom du chauffeur pour affichage
        FROM
            interventions i
        JOIN
            vehicules v ON i.id_vehicule = v.id_vehicule
        JOIN
            clients c ON v.id_client = c.id_client
        LEFT JOIN -- Utilisation de LEFT JOIN pour le chauffeur.
                  -- Cela permet d'afficher l'intervention même si le véhicule n'a pas de chauffeur associé.
                  -- La jointure est basée sur l'id_vehicule dans la table chauffeurs.
            chauffeurs ch ON v.id_vehicule = ch.id_vehicule
        WHERE
            i.id_intervention = ?
    ");
    $stmt->execute([$id_intervention_numeric]);
    $intervention = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$intervention) {
        die('Intervention introuvable.');
    }

} catch (PDOException $e) {
    error_log("Erreur PDO lors de la récupération des données pour le PDF : " . $e->getMessage());
    die('Erreur de base de données lors de la récupération des données.');
}

// Préparer les données pour l'affichage HTML
$date_declaration = (new DateTime($intervention['date_intervention']))->format('d/m/y');
$num_intervention = htmlspecialchars($id_intervention_formatted);
$demandeur = htmlspecialchars($intervention['nom_client']);
$num_vehicule = htmlspecialchars($intervention['immatriculation']);
$sig_vehicule = htmlspecialchars($intervention['sigle']);

$marque_vehicule = htmlspecialchars($intervention['marque']);
$kilometrage_intervention = htmlspecialchars($intervention['kilometrage_intervention']) . ' Km';
$origine_panne = htmlspecialchars($intervention['origine_panne']);
$rapport_intervention = htmlspecialchars($intervention['rapport_intervention'] ?? 'Aucun rapport détaillé.');
$date_creation_fiche = htmlspecialchars((new DateTime($intervention['date_creation']))->format('d/m/y'));
// Récupération du nom du chauffeur, avec une valeur par défaut si non trouvé
$nom_chauffeur = htmlspecialchars($intervention['nom_chauffeur'] ?? 'N/A'); 

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    <title>Fiche d'Intervention N°<?php echo $num_intervention; ?></title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 0; /* Réduit le padding général */
            background-color: #f4f4f4;
            display: flex;
            justify-content: center;
            align-items: flex-start;
            min-height: 100vh;
        }
        .fiche-container {
            width: 210mm; /* Largeur A4 */
            height: 297mm; /* Hauteur A4 Fixe pour forcer une seule page */
            overflow: hidden; /* Cache tout contenu qui dépasse */
            background-color: #fff;
            border: 1px solid #ccc;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
            padding: 15mm; /* Réduit le padding interne */
            box-sizing: border-box;
            font-size: 9.5pt; /* Légèrement réduit la taille de police générale */
            color: #333;
        }
        .header {
            text-align: center;
            margin-bottom: 8mm; /* Réduit la marge */
        }
        .header h1 {
            font-size: 16pt; /* Réduit la taille du titre */
            margin: 0;
            padding: 0;
            color: #000;
        }
        .info-grid {
            display: grid;
            grid-template-columns: repeat(6, 1fr); /* 6 colonnes pour les entêtes */
            gap: 2px 5mm; /* Réduit l'espacement */
            margin-bottom: 3mm; /* Réduit la marge */
            align-items: end;
        }
        .info-item {
            display: flex;
            flex-direction: column;
            gap: 1px; /* Réduit le gap */
        }
        .info-item label {
            font-weight: bold;
            font-size: 7pt; /* Réduit la taille de police */
            color: #555;
        }
        .info-item span {
            border-bottom: 1px solid #000;
            padding-bottom: 1px; /* Réduit le padding */
            min-height: 10px; /* Réduit la hauteur minimale */
            box-sizing: border-box;
        }
        .info-section-title {
            background-color: #e0e0e0;
            font-weight: bold;
            text-align: center;
            padding: 2px; /* Réduit le padding */
            border: 1px solid #ccc;
            margin-top: 3mm; /* Réduit la marge */
            margin-bottom: 2mm; /* Réduit la marge */
            font-size: 9.5pt; /* Réduit la taille de police */
        }
        .content-box {
            border: 1px solid #000;
            min-height: 15mm; /* Réduit la hauteur minimale */
            padding: 3px; /* Réduit le padding */
            box-sizing: border-box;
            margin-bottom: 3mm; /* Réduit la marge */
        }
        .text-area {
            min-height: 15mm; /* S'assure que le textarea est grand */
            border: none;
            width: 100%;
            padding: 0;
            margin: 0;
            resize: none;
            overflow: hidden;
            background-color: transparent;
        }
        .signature-grid {
            display: grid;
            grid-template-columns: 1fr 1fr; /* Deux colonnes pour les signatures */
            gap: 10mm; /* Réduit le gap */
            margin-top: 5mm; /* Réduit la marge */
        }
        .signature-item { /* Ajouté pour contrôler l'espacement de chaque bloc de signature */
            display: flex;
            flex-direction: column;
            align-items: center;
        }
        .signature-block {
            text-align: center;
            padding-top: 3mm; /* Réduit le padding */
            border-top: 1px solid #000;
            min-height: 10mm; /* Hauteur minimale pour la ligne de signature */
            width: 80%; /* Assure que la ligne de signature a une largeur raisonnable */
        }
        .signature-title {
            font-weight: bold;
            margin-bottom: 3px; /* Réduit la marge */
            font-size: 8.5pt;
        }
        .date-section {
            text-align: right;
            margin-top: 5mm; /* Réduit la marge */
            font-size: 8.5pt;
        }
        .date-section span {
            border-bottom: 1px solid #000;
            padding: 0 4mm 1px 4mm; /* Réduit le padding */
            display: inline-block;
        }

        /* Style pour le tableau de pièces */
        .parts-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1mm; /* Réduit fortement la marge */
            margin-bottom: 3mm; /* Réduit la marge */
        }
        .parts-table th, .parts-table td {
            border: 1px solid #000;
            padding: 1.5px; /* Réduit le padding */
            text-align: left;
            font-size: 7.5pt; /* Réduit la taille de police */
        }
        .parts-table th {
            background-color: #e0e0e0;
            text-align: center;
        }
        .col-span-full {
            grid-column: span 6;
        }
        .col-span-3 {
            grid-column: span 3;
        }
        .col-span-2 {
            grid-column: span 2;
        }
        .col-span-4 {
            grid-column: span 4;
        }

        /* Styles pour l'indicateur de chargement */
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            display: flex;
            justify-content: center;
            align-items: center;
            flex-direction: column;
            color: white;
            font-size: 1.5em;
            z-index: 9999;
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s, visibility 0.3s;
        }
        .loading-overlay.visible {
            opacity: 1;
            visibility: visible;
        }
        .spinner {
            border: 4px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top: 4px solid #fff;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin-bottom: 15px;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
         /* Styles des boutons d'action - toujours visibles sur la page HTML mais ignorés par html2canvas */
        .action-buttons {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            display: flex;
            gap: 15px;
            z-index: 1000;
        }
        .action-buttons button {
            padding: 10px 20px;
            font-size: 16px;
            cursor: pointer;
            background-color: #007BFF;
            color: white;
            border: none;
            border-radius: 5px;
            transition: background-color 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        .action-buttons button:hover {
            background-color: #0056b3;
        }
        .action-buttons button:nth-child(2) {
            background-color: #28a745;
        }
        .action-buttons button:nth-child(2):hover {
            background-color: #218838;
        }
        .action-buttons button:nth-child(3) {
            background-color: #6c757d;
        }
        .action-buttons button:nth-child(3):hover {
            background-color: #5a6268;
        }
    </style>
</head>
<body>
    <div class="fiche-container" id="fiche-intervention">
        <div class="header">
            <!-- Vous pouvez ajouter un logo ici si nécessaire -->
            <!-- <img src="img/logo_ase.png" alt="Logo" style="height: 50px; margin-bottom: 10px;"> -->
            <h1>FICHE D'INTERVENTION</h1>
        </div>

        <!-- Section EN-TÊTE -->
        <div class="info-grid">
            <div class="info-item col-span-2">
                <label>Date déclaration</label>
                <span><?php echo $date_declaration; ?></span>
            </div>
            <div class="info-item col-span-1">
                <label>Date début</label>
                <span></span>
            </div>
            <div class="info-item col-span-1">
                <label>Date fin</label>
                <span></span>
            </div>
            <div class="info-item col-span-1">
                <label>Durée</label>
                <span></span>
            </div>
            <div class="info-item col-span-2">
                <label>N°Bon/N°Devis</label>
                <span></span>
            </div>
            <div class="info-item col-span-2">
                <label>N° Intervention</label>
                <span><?php echo $num_intervention; ?></span>
            </div>
            <div class="info-item col-span-2">
                <label>N° Facture</label>
                <span></span>
            </div>

            <div class="info-item col-span-3">
                <label>DEMANDEUR</label>
                <span><?php echo $demandeur; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>Intervenant</label>
                <span></span>
            </div>

            <div class="info-item col-span-3">
                <label>N° VEHICULE</label>
                <span><?php echo $num_vehicule; ?>---<?php echo $sig_vehicule; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>Compteur Km</label>
                <span><?php echo $kilometrage_intervention; ?></span>
            </div>

            <div class="info-item col-span-3">
                <label>MARQUE</label>
                <span><?php echo $marque_vehicule; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>Compteur Horaire</label>
                <span></span>
            </div>

            <div class="info-item col-span-3">
                <label>CONDUCTEUR</label>
                <span><?php echo $nom_chauffeur; ?></span>
            </div>
            <div class="info-item col-span-3">
                <label>MODELE/TYPE</label>
                <span></span>
            </div>
        </div>

        <p style="margin-bottom: 3mm; font-size: 7pt;">
            Pour maintenir le véhicule en bon état de fonctionnement, pour qu'il soit toujours prêt à l'emploi, et pour détecter les causes éventuelles, causes d'usures prématurées des pièces, il faut utiliser le carburant et les lubrifiants préconisés, les liquides de qualité requise, et se conformer à l'ensemble des prescriptions de son entretien.
        </p>

        <!-- Section NATURE DE LA DEMANDE -->
        <div class="info-section-title">NATURE DE LA DEMANDE</div>
        <div class="info-grid" style="grid-template-columns: 4fr 2fr;">
            <div class="info-item">
                <label>Travaux demandés (travaux issus du diagnostic si cas de panne ou anomalie)</label>
                <span class="content-box" style="min-height: 25mm;"><?php echo nl2br($origine_panne); ?></span>
            </div>
            <div class="info-item">
                <label>Visa conducteur</label>
                <span class="content-box" style="min-height: 25mm;"></span>
            </div>
        </div>

        <!-- Section DÉTAILS DE L'INTERVENTION -->
        <div class="info-section-title">DETAILS DE L'INTERVENTION</div>
        <div class="info-grid" style="grid-template-columns: 4fr 2fr;">
            <div class="info-item">
                <label>DESCRIPTION</label>
                <span class="content-box" style="min-height: 25mm;"><?php echo nl2br($rapport_intervention); ?></span>
            </div>
            <div class="info-item">
                <label>Visa</label>
                <span class="content-box" style="min-height: 25mm;"></span>
            </div>
        </div>

        <!-- Section ACTIVITÉ(S) -->
        <div class="info-section-title">ACTIVITE(S)</div>
        <div class="content-box" style="min-height: 20mm;"></div>

        <!-- Section PRODUITS/PIECES CHANGEES -->
        <div class="info-section-title">PRODUITS/PIECES CHANGEES</div>
        <table class="parts-table">
            <thead>
                <tr>
                    <th style="width: 5%;">N°</th>
                    <th style="width: 25%;">Désignation</th>
                    <th style="width: 15%;">Référence</th>
                    <th style="width: 15%;">Numéro série</th>
                    <th style="width: 10%;">Quantité</th>
                    <th style="width: 10%;">PU</th>
                    <th style="width: 20%;">Provenance/Remarque</th>
                </tr>
            </thead>
            <tbody>
                <?php for ($i = 1; $i <= 7; $i++): ?>
                    <tr>
                        <td><?php echo $i; ?></td>
                        <td></td>
                        <td></td>
                        <td></td>
                        <td></td>
                        <td></td>
                        <td></td>
                    </tr>
                <?php endfor; ?>
            </tbody>
        </table>
        <div style="text-align: right; margin-bottom: 5mm; font-size: 9pt;">
            <b>Nombre article :</b> <span style="border-bottom: 1px solid #000; padding: 0 10mm;"></span>
        </div>


        <!-- Section Signatures -->
        <div class="signature-grid">
            <div class="signature-item">
                <div class="signature-title">Nom et Visa de Conducteur ou Client</div>
                <div class="signature-block"><?php echo $demandeur; ?></div>
            </div>
            <div class="signature-item">
                <div class="signature-title">Nom et visa du Responsable(s)</div>
                <div class="signature-block"></div>
            </div>
        </div>

        <div class="date-section">
            Fait à : <span></span> Le : <span><?php echo $date_creation_fiche; ?></span>
        </div>
    </div>

    <div class="action-buttons">
        <button onclick="downloadPdf()"><i class="fas fa-file-pdf"></i> Télécharger en PDF</button>
        <button onclick="window.close()"><i class="fas fa-times-circle"></i> Fermer</button>
    </div>

    <!-- Loading Overlay -->
    <div id="loadingOverlay" class="loading-overlay">
        <div class="spinner"></div>
        <p>Génération du PDF en cours...</p>
    </div>

    <script>
        window.jsPDF = window.jspdf.jsPDF; // Correctement initialiser jsPDF

        function downloadPdf() {
            const { jsPDF } = window.jspdf;
            const doc = new jsPDF('p', 'mm', 'a4');

            const element = document.getElementById('fiche-intervention');
            const loadingOverlay = document.getElementById('loadingOverlay');

            // Cacher les boutons et afficher l'overlay de chargement
            const actionButtons = document.querySelector('.action-buttons');
            if (actionButtons) {
                actionButtons.style.display = 'none';
            }
            loadingOverlay.classList.add('visible');


            // Options pour html2canvas. Utilisation d'une échelle de 1.5 pour un bon compromis qualité/performance
            const options = {
                scale: 1.5,
                useCORS: true,
                logging: false, // Désactive les logs de html2canvas dans la console
                letterRendering: true, // Tente d'améliorer le rendu du texte
                // Ignorer les éléments des boutons d'action lors de la capture
                ignoreElements: (element) => {
                    return element.classList.contains('action-buttons');
                }
            };

            html2canvas(element, options).then(canvas => {
                const imgData = canvas.toDataURL('image/png');
                const imgWidth = 210; // Largeur A4 en mm
                const pageHeight = 297; // Hauteur A4 en mm
                const imgHeight = canvas.height * imgWidth / canvas.width;
                let heightLeft = imgHeight;

                let position = 0;

                doc.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
                heightLeft -= pageHeight;

                // Gérer les pages multiples si le contenu dépasse une page A4
                // Dans ce cas, avec overflow:hidden sur fiche-container, il ne devrait y avoir qu'une seule "image"
                // Donc cette boucle ne devrait normalement s'exécuter qu'une fois.
                while (heightLeft > -10) { // La marge de -10 permet d'éviter les coupures à la fin de page
                    position = heightLeft - imgHeight;
                    doc.addPage();
                    doc.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
                    heightLeft -= pageHeight;
                }

                const filename = "fiche_intervention_<?php echo $num_intervention; ?>.pdf";
                doc.save(filename);

            }).catch(error => {
                console.error("Erreur lors de la capture du document avec html2canvas:", error);
                alert("Une erreur est survenue lors de la génération du PDF. Veuillez réessayer.");
            }).finally(() => {
                // Toujours masquer l'overlay et réafficher les boutons, même en cas d'erreur
                loadingOverlay.classList.remove('visible');
                if (actionButtons) {
                    actionButtons.style.display = 'flex';
                }
            });
        }
    </script>
</body>
</html>
