<?php
// ASE_Maintenance/generate_inventaire_pneu_fiche.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Inclusion de la bibliothèque Dompdf
require_once 'dompdf-3.1.0/dompdf/vendor/autoload.php';
use Dompdf\Dompdf;
use Dompdf\Options;

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Magasinier
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || $_SESSION['role'] !== 'Magasinier') {
    header('Location: index.php'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

// Récupération des filtres depuis l'URL
$filter_numero_serie = isset($_GET['filter_numero_serie']) ? trim($_GET['filter_numero_serie']) : '';
$filter_statut_pneu = isset($_GET['filter_statut_pneu']) ? trim($_GET['filter_statut_pneu']) : 'all';
$filter_modele = isset($_GET['filter_modele']) ? trim($_GET['filter_modele']) : 'all'; // id_pneu du modèle

$sql_where = " WHERE 1=1 ";
$params = [];

// Construction des clauses WHERE en fonction des filtres
if (!empty($filter_numero_serie)) {
    $sql_where .= " AND inv.numero_serie LIKE ?";
    $params[] = '%' . $filter_numero_serie . '%';
}
if ($filter_statut_pneu !== 'all') {
    $sql_where .= " AND inv.statut_pneu = ?";
    $params[] = $filter_statut_pneu;
}
if ($filter_modele !== 'all' && is_numeric($filter_modele)) {
    $sql_where .= " AND inv.id_pneu = ?";
    $params[] = (int)$filter_modele;
}

// Requête pour récupérer TOUS les pneus (sans pagination) selon les filtres
$sql_tires = "
    SELECT
        inv.id_inventaire_pneu,
        inv.numero_serie,
        inv.date_entree_systeme,
        inv.statut_pneu,
        inv.emplacement_actuel,
        inv.kilometrage_total_pneu,
        inv.profondeur_gomme_mm,
        inv.raison_hors_service,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil,
        mont.kilometrage_montage, -- Kilométrage du véhicule au moment du montage
        v.immatriculation AS vehicule_immatriculation, -- Immatriculation du véhicule si monté
        mont.position_pneu AS position_sur_vehicule -- Position sur le véhicule si monté
    FROM
        inventaire_pneu inv
    JOIN
        pneus p ON inv.id_pneu = p.id_pneu
    JOIN
        marques_pneu m ON p.id_marque = m.id_marque
    JOIN
        dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN
        profils_pneu pr ON p.id_profil = pr.id_profil
    LEFT JOIN -- Jointure pour récupérer les infos de montage si le pneu est monté
        montage_pneu_vehicule mont ON inv.id_inventaire_pneu = mont.id_inventaire_pneu AND mont.date_demontage IS NULL
    LEFT JOIN -- Jointure pour récupérer l'immatriculation du véhicule si monté
        vehicules v ON mont.id_vehicule = v.id_vehicule
    " . $sql_where . "
    ORDER BY inv.numero_serie ASC
";

$stmt_tires = $pdo->prepare($sql_tires);
$stmt_tires->execute($params);
$tires = $stmt_tires->fetchAll(PDO::FETCH_ASSOC);

// Définir les options de statut pour l'affichage des filtres
$statut_options = [
    'En stock' => 'En stock',
    'Monté' => 'Monté',
    'Sorti' => 'Sorti (hors gestion)',
    'Mis au rebut' => 'Mis au rebut',
    'En Transit' => 'En Transit',
    'Hors Service' => 'Hors Service (définitivement)'
];

// Récupérer les modèles de pneus pour l'affichage des filtres
$all_tire_models_map = [];
$stmt_all_tire_models = $pdo->query("
    SELECT
        p.id_pneu,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil
    FROM
        pneus p
    JOIN marques_pneu m ON p.id_marque = m.id_marque
    JOIN dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN profils_pneu pr ON p.id_profil = pr.id_profil
");
foreach ($stmt_all_tire_models->fetchAll(PDO::FETCH_ASSOC) as $model) {
    $all_tire_models_map[$model['id_pneu']] = $model['nom_modele'] . ' (' . $model['nom_marque'] . ' - ' . $model['valeur_dimension'] . ')';
}

// Préparer le logo en Base64 pour Dompdf
$logo_path = 'img/logo_ase.png';
$logo_base64 = '';
if (file_exists($logo_path)) {
    $type = pathinfo($logo_path, PATHINFO_EXTENSION);
    $data = file_get_contents($logo_path);
    $logo_base64 = 'data:image/' . $type . ';base64,' . base64_encode($data);
}

// Créer une instance de Dompdf
$options = new Options();
$options->set('isHtml5ParserEnabled', true);
$options->set('isRemoteEnabled', true);
$dompdf = new Dompdf($options);

ob_start();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Fiche d'Inventaire des Pneus</title>
    <style>
        html { height: 100%; }
        body {
            font-family: sans-serif;
            margin: 0;
            padding: 0;
            font-size: 10px;
            padding-bottom: 10mm;
        }

        @page { margin: 1cm; }

        .logo-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 70mm;
            height: 25mm;
            z-index: 1;
        }
        .logo-container img {
            max-width: 90%;
            height: auto;
            display: block;
        }

        .first-page-header-box {
            position: absolute;
            top: 0;
            right: 0;
            height: 20mm;
            width: 50mm;
            text-align: center;
            border: 1px solid black;
            padding: 8px 12px;
            font-size: 1.2em;
            font-weight: bold;
            box-sizing: border-box;
            margin-top: 10px;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 2;
        }
        .first-page-header-box p {
            margin: 2px 0;
            line-height: 1.4;
        }
        .first-page-header-box p:first-child {
            font-size: 1.3em;
            margin-bottom: 5px;
        }
        .document-container {
            width: 100%;
            position: relative;
            margin-top: 35mm;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 2mm;
            margin-bottom: 5mm;
            table-layout: fixed;
        }
        th, td {
            border: 1px solid black;
            padding: 8px 6px;
            text-align: left;
            font-size: 0.9em;
            vertical-align: top;
            word-wrap: break-word;
        }
        th {
            background-color: #e0e0e0;
            color: black;
            text-transform: uppercase;
            font-weight: bold;
            text-align: center;
        }
        thead { display: table-header-group; }

        .report-summary-box {
            margin-bottom: 10mm;
        }
        .report-summary-box strong {
            display: block;
            margin-bottom: 2mm;
        }
        .report-summary-box p {
            margin: 2px 0;
        }
        .text-info {
            width: 100%;
            display: table;
            margin-bottom: 5mm;
        }
        .text-info div {
            display: table-cell;
            width: 50%;
            vertical-align: top;
        }
        .no-data {
            text-align: center;
            margin-top: 20px;
            font-size: 1.2em;
        }

        .status-badge {
            padding: 3px 8px;
            border-radius: 4px;
            font-weight: bold;
            font-size: 0.6em;
            color: white;
            text-align: center;
            display: inline-block;
        }
        .status-En-stock { background-color: #28a745; }
        .status-Monté { background-color: #007bff; }
        .status-Sorti { background-color: #6c757d; }
        .status-Mis-au-rebut { background-color: #dc3545; }
        .status-En-Transit { background-color: #ffc107; color: #333; }
        .status-Hors-Service { background-color: #343a40; }

        #footer-content {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            width: 100%;
            text-align: center;
            font-size: 0.7em;
            color: black;
            border-top: 1px solid black;
            padding-top: 5px;
            box-sizing: border-box;
            line-height: 1.4;
        }
        #footer-content p {
            margin: 2px 0;
        }
    </style>
</head>
<body>
    <div class="logo-container">
        <?php if (!empty($logo_base64)): ?>
            <img src="<?= $logo_base64 ?>" alt="Logo ASE">
        <?php else: ?>
            <p style="color: red; font-size: 0.8em;">Logo non trouvé</p>
        <?php endif; ?>
    </div>

    <div class="first-page-header-box">
        <p>FICHE D'INVENTAIRE</p>
        <p>Généré le <?= date('d/m/Y') ?></p>
    </div>

    <div class="document-container">
        <div class="text-info">
            <div>
                <p><strong>Généré par :</strong> <?php echo htmlspecialchars($_SESSION['username'] ?? 'N/A'); ?> (<?php echo htmlspecialchars($_SESSION['role'] ?? 'N/A'); ?>)</p>
                <p><strong>Date de génération :</strong> <?php echo date('d/m/Y H:i:s'); ?></p>
            </div>
            <div>
                <p><strong>Modèle de Pneu :</strong> <?php echo $filter_modele !== 'all' ? htmlspecialchars($all_tire_models_map[$filter_modele]) : 'Tous'; ?></p>
            </div>
        </div>
        
        <?php if (empty($tires)): ?>
            <p class="no-data">Aucun pneu trouvé selon les critères de filtration.</p>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Modèle</th>
                        <th>Marque</th>
                        <th>Dimension</th>
                        <th>Km Montage Actuel</th>
                        <th>Kilométrage Total (km)</th>
                        <th>Raison HS</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($tires as $tire): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($tire['nom_modele']); ?></td>
                            <td><?php echo htmlspecialchars($tire['nom_marque']); ?></td>
                            <td><?php echo htmlspecialchars($tire['valeur_dimension']); ?></td>
                            <td><?php echo htmlspecialchars($tire['kilometrage_montage'] ? number_format($tire['kilometrage_montage'], 0, ',', ' ') . ' km' : 'N/A'); ?></td>
                            <td><?php echo number_format($tire['kilometrage_total_pneu'], 0, ',', ' '); ?></td>
                            <td><?php echo !empty($tire['raison_hors_service']) ? htmlspecialchars($tire['raison_hors_service']) : 'N/A'; ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>

    <div id="footer-content">
        <p><strong>AUTO SERVICE DE L'EST SARL</strong></p>
        <p>NIF: 3002015172 - STAT: 77100312000183</p>
        <p>Rue de commerce Toamasina 501 - Tél: +261 34 05 216 90 - Email: contact@ase.mg</p>
        <p>RIB BGFI: 00012 01250 41015401011 63 - RIB BFV: 00008 00490 02001012003 70</p>
    </div>

</body>
</html>
<?php
$html = ob_get_clean();
$dompdf->loadHtml($html);
$dompdf->setPaper('A4', 'portrait');
$dompdf->render();

$filename = "Fiche_Inventaire_Pneu_" . date('Ymd_His') . ".pdf";
$dompdf->stream($filename, ["Attachment" => true]);
exit();
?>