<?php
// ASE_Maintenance/generate_bon_sortie_pneu_pdf.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Inclusion de la bibliothèque Dompdf
require_once 'dompdf-3.1.0/dompdf/vendor/autoload.php';
use Dompdf\Dompdf;
use Dompdf\Options;

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Magasinier
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || $_SESSION['role'] !== 'Magasinier') {
    header('Location: index.php'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

// Récupération des filtres de date depuis l'URL
$date_debut = isset($_GET['date_debut']) ? $_GET['date_debut'] : '';
$date_fin = isset($_GET['date_fin']) ? $_GET['date_fin'] : '';

$grouped_sorties = [];
$error_message = null;

// Validation des dates
if (empty($date_debut) || empty($date_fin)) {
    $error_message = "Veuillez spécifier une date de début et une date de fin pour générer le bon de sortie.";
} else {
    // Ajouter l'heure de début et de fin pour couvrir toute la journée
    $date_debut_datetime = $date_debut . ' 00:00:00';
    $date_fin_datetime = $date_fin . ' 23:59:59';

    $sql_where = " WHERE s.date_sortie BETWEEN ? AND ? ";
    $params = [$date_debut_datetime, $date_fin_datetime];

    // Requête pour récupérer les détails des sorties de pneus
    $sql_sorties = "
        SELECT
            s.id_sortie_pneu,
            s.date_sortie,
            s.raison_sortie,
            u.nom_utilisateur AS nom_magasinier,
            inv.numero_serie,
            p.nom_modele,
            m.nom_marque,
            d.valeur_dimension,
            pr.nom_profil,
            inv.statut_pneu AS statut_final_pneu,
            inv.emplacement_actuel,
            inv.kilometrage_total_pneu,
            inv.profondeur_gomme_mm,
            COALESCE(
                (SELECT mont.kilometrage_demontage
                 FROM montage_pneu_vehicule mont
                 WHERE mont.id_inventaire_pneu = inv.id_inventaire_pneu
                 AND mont.date_demontage = s.date_sortie
                 ORDER BY mont.date_demontage DESC LIMIT 1),
                0
            ) AS kilometrage_au_demontage_sortie
        FROM
            sorties_pneu s
        JOIN
            details_sortie_pneu ds ON s.id_sortie_pneu = ds.id_sortie_pneu
        JOIN
            inventaire_pneu inv ON ds.id_inventaire_pneu = inv.id_inventaire_pneu
        JOIN
            pneus p ON inv.id_pneu = p.id_pneu
        JOIN
            marques_pneu m ON p.id_marque = m.id_marque
        JOIN
            dimensions_pneu d ON p.id_dimension = d.id_dimension
        JOIN
            profils_pneu pr ON p.id_profil = pr.id_profil
        JOIN
            utilisateurs u ON s.id_magasinier = u.id_utilisateur
        " . $sql_where . "
        ORDER BY s.date_sortie DESC, inv.numero_serie ASC
    ";

    $stmt_sorties = $pdo->prepare($sql_sorties);
    $stmt_sorties->execute($params);
    $sorties_data = $stmt_sorties->fetchAll(PDO::FETCH_ASSOC);

    // Groupement des données par bon de sortie (id_sortie_pneu)
    foreach ($sorties_data as $row) {
        $id_sortie = $row['id_sortie_pneu'];
        if (!isset($grouped_sorties[$id_sortie])) {
            $grouped_sorties[$id_sortie] = [
                'id_sortie_pneu' => $row['id_sortie_pneu'],
                'date_sortie' => $row['date_sortie'],
                'raison_sortie' => $row['raison_sortie'],
                'nom_magasinier' => $row['nom_magasinier'],
                'pneus' => []
            ];
        }
        $grouped_sorties[$id_sortie]['pneus'][] = $row;
    }
}

// Informations de l'entreprise (fixes) pour l'en-tête et le pied de page
$nom_entreprise = "AUTO SERVICE DE L'EST SARL";
$adresse_entreprise = "Rue de commerce Toamasina 501";
$nif_entreprise = "3002015172";
$stat_entreprise = "77100312000183";
$tel_entreprise = "+261 34 05 216 90";
$email_entreprise = "contact@ase.mg";
$rib_bgfi = "00012 01250 41015401011 63";
$rib_bfv = "00008 00490 02001012003 70";

// Préparer le logo en Base64 pour Dompdf
$logo_path = 'img/logo_ase.png';
$logo_base64 = '';
if (file_exists($logo_path)) {
    $type = pathinfo($logo_path, PATHINFO_EXTENSION);
    $data = file_get_contents($logo_path);
    $logo_base64 = 'data:image/' . $type . ';base64,' . base64_encode($data);
}

// Créer une instance de Dompdf
$options = new Options();
$options->set('isHtml5ParserEnabled', true);
$options->set('isRemoteEnabled', true);
$dompdf = new Dompdf($options);

ob_start();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Bon de Sortie des Pneus</title>
    <style>
        html { height: 100%; }
        body {
            font-family: sans-serif;
            margin: 0;
            padding: 0;
            font-size: 10px;
            padding-bottom: 10mm;
        }

        @page { margin: 1cm; }

        .logo-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 70mm;
            height: 25mm;
            z-index: 1;
        }
        .logo-container img {
            max-width: 90%;
            height: auto;
            display: block;
        }

        .first-page-header-box {
            position: absolute;
            top: 0;
            right: 0;
            height: 20mm;
            width: 50mm;
            text-align: center;
            border: 1px solid black;
            padding: 8px 12px;
            font-size: 1.2em;
            font-weight: bold;
            box-sizing: border-box;
            margin-top: 10px;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 2;
        }
        .first-page-header-box p {
            margin: 2px 0;
            line-height: 1.4;
        }
        .first-page-header-box p:first-child {
            font-size: 1.3em;
            margin-bottom: 5px;
        }
        .document-container {
            width: 100%;
            position: relative;
            margin-top: 35mm;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 2mm;
            margin-bottom: 5mm;
        }
        th, td {
            border: 1px solid black;
            padding: 8px 6px;
            text-align: left;
            font-size: 0.9em;
            vertical-align: top;
            word-wrap: break-word;
        }
        th {
            background-color: #e0e0e0;
            color: black;
            text-transform: uppercase;
            font-weight: bold;
            text-align: center;
        }
        thead { display: table-header-group; }

        .bon-sortie-header {
            margin-bottom: 15px;
            border-bottom: 1px solid #ccc;
            padding-bottom: 10px;
            page-break-after: avoid; /* Empêche une coupure de page juste après le titre */
        }
        .bon-sortie-header h2 {
            margin: 0;
            font-size: 1.2em;
        }
        .bon-sortie-header p {
            margin: 2px 0;
            font-size: 0.9em;
        }
        .no-data {
            text-align: center;
            margin-top: 20px;
            font-size: 1.2em;
        }

        #footer-content {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            width: 100%;
            text-align: center;
            font-size: 0.7em;
            color: black;
            border-top: 1px solid black;
            padding-top: 5px;
            box-sizing: border-box;
            line-height: 1.4;
        }
        #footer-content p {
            margin: 2px 0;
        }
    </style>
</head>
<body>
    <div class="logo-container">
        <?php if (!empty($logo_base64)): ?>
            <img src="<?= $logo_base64 ?>" alt="Logo ASE">
        <?php else: ?>
            <p style="color: red; font-size: 0.8em;">Logo non trouvé</p>
        <?php endif; ?>
    </div>

    <div class="first-page-header-box">
        <p>BON DE SORTIE</p>
        <p>Pneus</p>
    </div>

    <div class="document-container">
        <div style="margin-bottom: 15px; font-size: 1.1em;">
            <p><strong>Période du :</strong> <?php echo htmlspecialchars(date('d/m/Y', strtotime($date_debut))); ?></p>
            <p><strong>Au :</strong> <?php echo htmlspecialchars(date('d/m/Y', strtotime($date_fin))); ?></p>
        </div>
        
        <?php if ($error_message): ?>
            <p class="no-data"><?php echo htmlspecialchars($error_message); ?></p>
        <?php elseif (empty($grouped_sorties)): ?>
            <p class="no-data">Aucun bon de sortie trouvé pour la période spécifiée.</p>
        <?php else: ?>
            <?php foreach ($grouped_sorties as $id_sortie => $sortie): ?>
                <div class="bon-sortie-header">
                    <h2>Bon de Sortie #<?php echo htmlspecialchars($id_sortie); ?></h2>
                    <p><strong>Date de sortie :</strong> <?php echo htmlspecialchars(date('d/m/Y H:i', strtotime($sortie['date_sortie']))); ?></p>
                    <p><strong>Raison :</strong> <?php echo htmlspecialchars($sortie['raison_sortie']); ?></p>
                    <p><strong>Magasinier :</strong> <?php echo htmlspecialchars($sortie['nom_magasinier']); ?></p>
                </div>
                
                <table>
                    <thead>
                        <tr>
                            <th>N° Série</th>
                            <th>Modèle</th>
                            <th>Emplacement Final</th>
                            <th>Km au démontage</th>
                            <th>Km Total</th>
                            <th>Gomme (mm)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($sortie['pneus'] as $item): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($item['numero_serie']); ?></td>
                                <td><?php echo htmlspecialchars($item['nom_modele'] . ' (' . $item['nom_marque'] . ' - ' . $item['valeur_dimension'] . ')'); ?></td>
                                <td><?php echo htmlspecialchars($item['emplacement_actuel']); ?></td>
                                <td><?php echo number_format($item['kilometrage_au_demontage_sortie'], 0, ',', ' '); ?> km</td>
                                <td><?php echo number_format($item['kilometrage_total_pneu'], 0, ',', ' '); ?> km</td>
                                <td><?php echo !empty($item['profondeur_gomme_mm']) ? htmlspecialchars($item['profondeur_gomme_mm']) : 'N/A'; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>

    <div id="footer-content">
        <p><strong><?= htmlspecialchars($nom_entreprise) ?></strong></p>
        <p>NIF: <?= htmlspecialchars($nif_entreprise); ?> - STAT: <?= htmlspecialchars($stat_entreprise); ?></p>
        <p><?= htmlspecialchars($adresse_entreprise); ?> - Tél: <?= htmlspecialchars($tel_entreprise); ?> - Email: <?= htmlspecialchars($email_entreprise); ?></p>
        <p>RIB BGFI: <?= htmlspecialchars($rib_bgfi); ?> - RIB BFV: <?= htmlspecialchars($rib_bfv); ?></p>
    </div>

</body>
</html>
<?php
$html = ob_get_clean();
$dompdf->loadHtml($html);
$dompdf->setPaper('A4', 'portrait');
$dompdf->render();

$filename = "Bon_de_Sortie_Pneu_" . date('Ymd_His') . ".pdf";
$dompdf->stream($filename, ["Attachment" => true]);
exit();
?>