<?php
// ASE_Maintenance/commande_pneu.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Vérification de l'authentification et du rôle Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

$records_per_page = 5;

// Endpoint pour récupérer les prix d'un modèle de pneu (via AJAX)
if (isset($_GET['action']) && $_GET['action'] === 'get_pneu_price' && isset($_GET['id_pneu'])) {
    header('Content-Type: application/json');
    $id_pneu = filter_var($_GET['id_pneu'], FILTER_VALIDATE_INT);

    if ($id_pneu) {
        $stmt = $pdo->prepare("SELECT prix_achat_ht, tva_applicable FROM pneus WHERE id_pneu = ?");
        $stmt->execute([$id_pneu]);
        $pneu_data = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($pneu_data) {
            echo json_encode(['success' => true, 'prix_achat_ht' => $pneu_data['prix_achat_ht'], 'tva_applicable' => $pneu_data['tva_applicable']]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Modèle de pneu non trouvé.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID de pneu invalide.']);
    }
    exit();
}

// Endpoint pour récupérer les détails d'une commande (via AJAX pour la modification)
// Le paramètre attendu est maintenant id_commande_pneu
if (isset($_GET['action']) && $_GET['action'] === 'get_commande_details' && isset($_GET['id_commande_pneu'])) {
    header('Content-Type: application/json');
    $id_commande_pneu = filter_var($_GET['id_commande_pneu'], FILTER_VALIDATE_INT);

    if ($id_commande_pneu) {
        // La requête sélectionne sur id_commande_pneu
        $commande_stmt = $pdo->prepare("SELECT * FROM commandes_pneu_appro WHERE id_commande_pneu = ?");
        $commande_stmt->execute([$id_commande_pneu]);
        $commande = $commande_stmt->fetch(PDO::FETCH_ASSOC);

        if ($commande) {
            $details_stmt = $pdo->prepare("
                SELECT
                    dc.id_detail_commande,
                    dc.id_pneu,
                    p.nom_modele,
                    m.nom_marque,
                    d.valeur_dimension,
                    pr.nom_profil,
                    dc.quantite_commandee,
                    dc.prix_ht_unitaire,
                    dc.tva_unitaire,
                    (dc.prix_ht_unitaire * (1 + dc.tva_unitaire / 100)) AS prix_unitaire_ttc
                FROM
                    details_commande_pneu_appro dc
                JOIN
                    pneus p ON dc.id_pneu = p.id_pneu
                JOIN
                    marques_pneu m ON p.id_marque = m.id_marque
                JOIN
                    dimensions_pneu d ON p.id_dimension = d.id_dimension
                JOIN
                    profils_pneu pr ON p.id_profil = pr.id_profil
                WHERE
                    dc.id_commande_pneu = ?
            ");
            $details_stmt->execute([$id_commande_pneu]);
            $details = $details_stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode(['success' => true, 'commande' => $commande, 'details' => $details]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Modèle de pneu non trouvé.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID de commande invalide.']);
    }
    exit();
}

// Traitement des requêtes POST (Création/Modification/Clôture)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'cloturer_livraison') {
        $id_commande_pneu = filter_var($_POST['id_commande_pneu'], FILTER_VALIDATE_INT);
        if ($id_commande_pneu) {
            try {
                $pdo->beginTransaction();
                // La mise à jour se fait sur id_commande_pneu
                $stmt = $pdo->prepare("UPDATE commandes_pneu_appro SET statut_livraison = 'Livrée', statut_commande = 'Terminée' WHERE id_commande_pneu = ? AND statut_livraison = 'Partiellement livrée'");
                $stmt->execute([$id_commande_pneu]);
                
                // Pour la clôture, on ajuste la quantité_commandee à la quantité réellement reçue (qui n'est pas encore implémentée)
                // Pour l'instant, on considère que la réception est égale à la commande pour la clôture simple
                // Ce bloc sera ajusté avec l'implémentation de reception_pneu.php
                // Par exemple, on pourrait mettre à jour le stock ici en utilisant les quantités reçues de la table de réception.
                
                $_SESSION['message'] = "Livraison de la commande N°" . $id_commande_pneu . " clôturée avec succès.";
                $_SESSION['message_type'] = 'success';
                $pdo->commit();
            } catch (PDOException $e) {
                $pdo->rollBack();
                $_SESSION['message'] = "Erreur lors de la clôture de la livraison : " . $e->getMessage();
                $_SESSION['message_type'] = 'error';
            }
        } else {
            $_SESSION['message'] = "ID de commande invalide pour la clôture.";
            $_SESSION['message_type'] = 'error';
        }
        header("Location: commande_pneu.php");
        exit();
    }
    
    // Traitement de l'ajout/modification de commande
    if (isset($_POST['command_lines'])) {
        // Le paramètre id_commande_pneu est maintenant attendu dans le POST
        $id_commande_pneu = isset($_POST['id_commande_pneu']) ? filter_var($_POST['id_commande_pneu'], FILTER_VALIDATE_INT) : null;
        $command_lines = json_decode($_POST['command_lines'], true); // Décode les lignes de commande
        $commentaire = isset($_POST['commentaire']) ? trim($_POST['commentaire']) : null;


        if (empty($command_lines)) {
            $_SESSION['message'] = "Veuillez ajouter au moins une ligne à la commande.";
            $_SESSION['message_type'] = 'error';
            header("Location: commande_pneu.php");
            exit();
        }

        try {
            $pdo->beginTransaction();

            $total_ht = 0;
            $total_tva = 0;
            $total_ttc = 0;

            // Récupérer les prix réels depuis la base de données pour chaque pneu
            $pneu_prices = [];
            $stmt_prices = $pdo->prepare("SELECT id_pneu, prix_achat_ht, tva_applicable FROM pneus WHERE id_pneu IN (" . implode(',', array_fill(0, count($command_lines), '?')) . ")");
            $pneu_ids_for_query = array_column($command_lines, 'id_pneu');
            $stmt_prices->execute($pneu_ids_for_query);
            foreach ($stmt_prices->fetchAll(PDO::FETCH_ASSOC) as $p) {
                $pneu_prices[$p['id_pneu']] = $p;
            }

            if ($id_commande_pneu) {
                // MODIFICATION DE COMMANDE EXISTANTE
                // La mise à jour se fait sur id_commande_pneu
                $stmt_update_commande = $pdo->prepare("UPDATE commandes_pneu_appro SET date_commande = NOW(), id_magasinier = ?, statut_commande = 'En attente', statut_livraison = 'En attente', commentaire = ?, total_ht = ?, total_tva = ?, total_ttc = ? WHERE id_commande_pneu = ?");
                // La suppression des détails se fait sur id_commande_pneu
                $stmt_delete_details = $pdo->prepare("DELETE FROM details_commande_pneu_appro WHERE id_commande_pneu = ?");
                $stmt_delete_details->execute([$id_commande_pneu]);
                
            } else {
                // NOUVELLE COMMANDE
                $stmt_insert_commande = $pdo->prepare("INSERT INTO commandes_pneu_appro (date_commande, id_magasinier, statut_commande, statut_livraison, commentaire, total_ht, total_tva, total_ttc) VALUES (NOW(), ?, 'En attente', 'En attente', ?, ?, ?, ?)");
            }

            $id_magasinier = $_SESSION['user_id'];

            foreach ($command_lines as $line) {
                $id_pneu = filter_var($line['id_pneu'], FILTER_VALIDATE_INT);
                $quantite_commandee = filter_var($line['quantite_commandee'], FILTER_VALIDATE_INT);

                if (!$id_pneu || !$quantite_commandee || $quantite_commandee <= 0) {
                    throw new Exception("Données de ligne de commande invalides.");
                }

                $pneu_data = $pneu_prices[$id_pneu] ?? null;
                if (!$pneu_data) {
                    throw new Exception("Prix du modèle de pneu non trouvé pour l'ID: " . $id_pneu);
                }

                $prix_unitaire_ht = $pneu_data['prix_achat_ht'];
                $tva_applicable = $pneu_data['tva_applicable'];
                $tva_decimal = $tva_applicable / 100;
                $prix_unitaire_ttc = $prix_unitaire_ht * (1 + $tva_decimal);

                $total_ht += $prix_unitaire_ht * $quantite_commandee;
                $total_tva += ($prix_unitaire_ht * $tva_decimal) * $quantite_commandee;
                $total_ttc += $prix_unitaire_ttc * $quantite_commandee;
            }

            // Exécuter l'insertion/mise à jour de la commande principale
            if ($id_commande_pneu) {
                $stmt_update_commande->execute([$id_magasinier, $commentaire, $total_ht, $total_tva, $total_ttc, $id_commande_pneu]);
                $_SESSION['message'] = "Commande N°" . $id_commande_pneu . " modifiée avec succès !";
                $_SESSION['message_type'] = 'success';
            } else {
                $stmt_insert_commande->execute([$id_magasinier, $commentaire, $total_ht, $total_tva, $total_ttc]);
                $id_commande_pneu = $pdo->lastInsertId(); // Récupère le dernier ID inséré (qui est id_commande_pneu)
                $_SESSION['message'] = "Commande N°" . $id_commande_pneu . " créée avec succès !";
                $_SESSION['message_type'] = 'success';
            }

            // Insérer/Mettre à jour les détails de la commande
            // La colonne dans details_commande_pneu_appro est id_commande_pneu
            $stmt_insert_detail = $pdo->prepare("INSERT INTO details_commande_pneu_appro (id_commande_pneu, id_pneu, quantite_commandee, prix_ht_unitaire, tva_unitaire) VALUES (?, ?, ?, ?, ?)");
            foreach ($command_lines as $line) {
                $id_pneu = filter_var($line['id_pneu'], FILTER_VALIDATE_INT);
                $quantite_commandee = filter_var($line['quantite_commandee'], FILTER_VALIDATE_INT);
                
                $pneu_data = $pneu_prices[$id_pneu] ?? null;
                $prix_unitaire_ht_detail = $pneu_data['prix_achat_ht'];
                $tva_applicable_detail = $pneu_data['tva_applicable'];

                $stmt_insert_detail->execute([$id_commande_pneu, $id_pneu, $quantite_commandee, $prix_unitaire_ht_detail, $tva_applicable_detail]);
            }

            $pdo->commit();

        } catch (Exception $e) {
            $pdo->rollBack();
            $_SESSION['message'] = "Erreur : " . $e->getMessage();
            $_SESSION['message_type'] = 'error';
        } catch (PDOException $e) {
            $pdo->rollBack();
            $_SESSION['message'] = "Erreur de base de données : " . $e->getMessage();
            $_SESSION['message_type'] = 'error';
        }
        header("Location: commande_pneu.php");
        exit();
    }
}

// Récupération des commandes existantes
$current_page_commandes = isset($_GET['page_commandes']) && is_numeric($_GET['page_commandes']) ? (int)$_GET['page_commandes'] : 1;
$offset_commandes = ($current_page_commandes - 1) * $records_per_page;

$filter_status = isset($_GET['filter_status']) ? $_GET['filter_status'] : '';

$sql_where = '';
$params = [];
if (!empty($filter_status) && $filter_status !== 'all') {
    $sql_where = " WHERE statut_livraison = ?";
    $params[] = $filter_status;
}

// Compter le total des commandes en utilisant le bon nom de colonne
$total_commandes_query = "SELECT COUNT(*) FROM commandes_pneu_appro" . $sql_where;
$stmt_total_commandes = $pdo->prepare($total_commandes_query);
$stmt_total_commandes->execute($params);
$total_commandes = $stmt_total_commandes->fetchColumn();
$total_pages_commandes = ceil($total_commandes / $records_per_page);

$sql_commandes = "
    SELECT
        c.id_commande_pneu,
        c.date_commande,
        u.nom_utilisateur AS nom_magasinier,
        c.statut_commande,
        c.statut_livraison,
        c.commentaire
    FROM
        commandes_pneu_appro c
    JOIN
        utilisateurs u ON c.id_magasinier = u.id_utilisateur
    " . $sql_where . "
    ORDER BY c.date_commande DESC
    LIMIT ? OFFSET ?
";
$params[] = $records_per_page;
$params[] = $offset_commandes;

$stmt_commandes = $pdo->prepare($sql_commandes);
$stmt_commandes->execute($params);
$commandes = $stmt_commandes->fetchAll(PDO::FETCH_ASSOC);

// Récupérer tous les modèles de pneus pour le formulaire (ID, Nom, Marque, Dimension, Profil)
$all_pneus_models_stmt = $pdo->query("
    SELECT
        p.id_pneu,
        p.nom_modele,
        m.nom_marque,
        d.valeur_dimension,
        pr.nom_profil
    FROM
        pneus p
    JOIN
        marques_pneu m ON p.id_marque = m.id_marque
    JOIN
        dimensions_pneu d ON p.id_dimension = d.id_dimension
    JOIN
        profils_pneu pr ON p.id_profil = pr.id_profil
    ORDER BY p.nom_modele
");
$all_pneus_models = $all_pneus_models_stmt->fetchAll(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Commande Pneus - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Maintenir la taille pour le texte du lien */
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 0.9em; /* Maintenir la taille pour les icônes de lien */
        }
        
        /* Style spécifique pour l'icône de déconnexion */
        .navbar-nav .fa-power-off {
            font-size: 1.5em !important; /* Augmenter la taille pour l'icône de déconnexion */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Styles pour le menu déroulant (dropdown) */
        .dropdown-menu {
            display: none; /* Caché par défaut, sera géré par JS */
            position: absolute;
            background-color: var(--navbar-bg);
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.6);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: calc(100% + 5px); /* Position sous l'élément parent */
            left: 0;
            list-style: none;
        }

        .dropdown-menu li {
            margin: 0;
        }

        .dropdown-menu a {
            color: var(--text-color);
            padding: 12px 16px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: 0.95em;
        }

        .dropdown-menu a:hover {
            background-color: rgba(255, 218, 71, 0.1);
            color: var(--accent-color);
        }

        .dropdown.active .dropdown-menu {
            display: block;
        }


        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }
        h3.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 8px;
            margin-top: 30px;
            margin-bottom: 20px;
            font-size: 1.8em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group select,
        .form-group textarea {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-group input[type="text"]:focus,
        .form-group input[type="number"]:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }
        .form-group input[type="text"][readonly] {
            background-color: rgba(var(--input-bg), 0.5);
            cursor: not-allowed;
            opacity: 0.8;
        }

        /* Buttons styles */
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        .btn-icon {
            padding: 8px;
            width: 40px;
            height: 40px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i { margin-right: 0 !important; }

        /* --- Table styles --- */
        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 30px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 600px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 12px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        tr:hover {
            background-color: #3a3a3a;
            transition: background-color 0.2s ease;
        }

        .table-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-start;
            align-items: center;
        }

        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
        }

        .pagination a,
        .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination span.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title, h3.section-title { font-size: 2em; }
            .form-group input[type="text"],
            .form-group input[type="number"],
            .form-group select,
            .form-group textarea {
                width: 100%;
            }
            .dropdown-menu {
                position: static;
                box-shadow: none;
                background-color: transparent;
                width: 100%;
                padding-left: 20px;
                border-radius: 0;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title, h3.section-title { font-size: 1.8em; margin-top: 30px; }
            th, td { padding: 8px; font-size: 0.85em; }
            table { min-width: unset; }
            .btn { width: 100%; justify-content: center; margin-bottom: 12px; }
            .pagination {
                flex-wrap: wrap;
                gap: 5px;
            }
            .pagination a, .pagination span {
                padding: 8px 12px;
                font-size: 0.9em;
            }
            .dropdown > a {
                cursor: pointer;
            }
            .button-group-modal {
                flex-direction: column;
                gap: 10px;
            }
            /* Formulaire 2 champs par ligne sur mobile -> 1 champ par ligne */
            .form-row-2-cols > .form-group {
                width: 100% !important;
                padding-left: 0 !important;
                padding-right: 0 !important;
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title, h3.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
            .form-section, .list-section { padding: 20px; }
            .btn { font-size: 1em; padding: 10px 20px; }
        }

        /* --- MODAL STYLES --- */
        .modal1 {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.7);
            justify-content: center;
            align-items: center;
            -webkit-animation: fadeIn 0.3s forwards;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background-color: var(--form-bg);
            margin: auto;
            padding: 25px 30px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 90%;
            max-width: 900px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            transition: transform 0.2s ease-out;
            -webkit-animation: slideIn 0.3s forwards;
            animation: slideIn 0.3s forwards;
        }

        .modal1.open {
            display: flex;
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        .close-button {
            color: var(--text-color);
            position: absolute;
            top: 15px;
            right: 25px;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close-button:hover,
        .close-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        @-webkit-keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @-webkit-keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @-webkit-keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @-webkit-keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        @keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        /* End MODAL STYLES */

        /* Styles spécifiques pour les lignes de commande dynamiques */
        .commande-line-item {
            display: flex;
            flex-wrap: wrap;
            gap: 10px; /* Espace entre les éléments de la ligne */
            background-color: var(--input-bg);
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 15px;
            align-items: flex-end;
            position: relative; /* Pour le bouton de suppression */
        }
        .commande-line-item .form-group {
            flex: 1 1 calc(50% - 5px); /* Deux colonnes */
            margin-bottom: 0; /* Gérer les marges par le gap */
        }
        .commande-line-item .form-group select,
        .commande-line-item .form-group input {
            width: 100%;
        }
        .remove-line-btn {
            position: absolute;
            top: 10px;
            right: 10px;
            background-color: var(--btn-danger-bg);
            color: white;
            border: none;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 1.2em;
            display: flex;
            justify-content: center;
            align-items: center;
            cursor: pointer;
            transition: background-color 0.2s ease;
        }
        .remove-line-btn:hover {
            background-color: var(--btn-danger-hover);
        }
        .commande-line-item .form-group.full-width { /* Pour les éléments qui prennent toute la largeur dans une ligne */
            flex: 1 1 100%;
        }

        /* Style pour les détails de commande déroulants */
        .details-row-content {
            max-height: 0; /* Initialement caché */
            overflow: hidden;
            transition: max-height 0.4s ease-in-out; /* Animation de transition */
            width: 100%;
            padding: 0 15px; /* Padding à 0 initialement pour la transition */
            background-color: #333;
            border-top: 1px solid var(--border-color);
            box-sizing: border-box;
        }
        /* Classe pour afficher les détails */
        .details-row-content.active {
            max-height: 1000px; /* Une valeur suffisamment grande pour contenir le contenu */
            padding: 15px; /* Le padding s'applique lorsque la section est active */
        }
        .details-row-content table {
            width: 100%;
            background-color: #3c3c3c;
            border-radius: 5px;
        }
        .details-row-content th,
        .details-row-content td {
            padding: 8px;
            font-size: 0.85em;
            border: 1px solid #505050;
        }
        .details-row-content th {
            background-color: #2a2a2a;
        }
        .details-row-content tr:nth-child(even) {
            background-color: #333;
        }

        /* Styles pour les statuts */
        .status-badge {
            padding: 5px 10px;
            border-radius: 5px;
            font-weight: bold;
            font-size: 0.9em;
            color: #333; /* Texte sombre par défaut pour les badges clairs */
        }

        /* Statut Commande */
        .status-En-attente { background-color: #ffc107; } /* Jaune - Warning */
        .status-Terminee { background-color: #28a745; color: white; } /* Vert - Success */
        .status-Annulee { background-color: #dc3545; color: white; } /* Rouge - Danger */

        /* Statut Livraison */
        .status-En-attente-livraison { background-color: #17a2b8; color: white; } /* Bleu clair - Info */
        .status-Partiellement-livree { background-color: #fd7e14; color: white; } /* Orange - Custom */
        .status-Livree { background-color: #28a745; color: white; } /* Vert - Success */
        .commentaire-details {
            white-space: pre-wrap;
            background-color: var(--input-bg);
            padding: 10px;
            border-radius: 5px;
            margin-top: 10px;
            border: 1px solid var(--input-border);
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande"><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li class="dropdown active"> <a href="#" class="dropdown-toggle" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-car"></i> Gestion Pneumatique <span>--</span><i class="fas fa-caret-down"></i> </a>
                <ul class="dropdown-menu">
                    <li><a href="pneu"><i class="fas fa-car"></i> Modèle Pneumatique </a></li> 
                    <li><a href="commande_pneu"><i class="fas fa-shopping-cart"></i> Commander Pneus</a></li>
                    <li><a href="reception_pneu"><i class="fas fa-truck-loading"></i> Réception Pneus</a></li>
                    <li><a href="inventaire_pneu"><i class="fas fa-warehouse"></i> Inventaire Pneus</a></li>
                    <li><a href="montage_pneu"><i class="fas fa-wrench"></i> Montage/Démontage</a></li> </ul>
            </li>
            <li><a href="php/authentification.php?action=logout"> <i class="fa-solid fa-power-off"></i></a></li> </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1 id="page-title">Gestion des Commandes de Pneus</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="form-section">
            <h2 class="section-title">Créer / Modifier une Commande de Pneus</h2>
            <form action="commande_pneu.php" method="POST" id="commandeForm">
                <input type="hidden" name="id_commande_pneu" id="id_commande_input" value="">
                <input type="hidden" name="command_lines" id="command_lines_input">

                <div class="form-group">
                    <label for="commentaire">Commentaire :</label>
                    <textarea name="commentaire" id="commentaire" rows="3" placeholder="Ajouter un commentaire pour cette commande..."></textarea>
                </div>

                <div id="commande_lines_container">
                    </div>

                <div class="flex justify-between items-center mt-6">
                    <button type="button" class="btn btn-info" id="add_line_btn"><i class="fas fa-plus"></i> Ajouter une ligne de pneu</button>
                    <div>
                        <button type="submit" class="btn btn-success" id="submit_commande_btn"><i class="fas fa-file-circle-plus"></i> Créer la commande</button>
                        <button type="button" class="btn btn-danger" id="cancel_edit_commande_btn" style="display:none;"><i class="fas fa-ban"></i> Annuler Modification</button>
                    </div>
                </div>
            </form>
        </div>

        <hr style="margin: 40px 0; border-top: 1px solid var(--border-color);">

        <div class="list-section">
            <h2 class="section-title">Commandes de Pneus Existantes</h2>
            <div class="mb-4 flex items-center gap-4">
                <label for="filter_status" class="font-bold">Filtrer par statut de livraison :</label>
                <select id="filter_status" class="p-2 rounded bg-gray-700 border border-gray-600 text-white">
                    <option value="all" <?php echo ($filter_status === 'all' || empty($filter_status)) ? 'selected' : ''; ?>>Toutes</option>
                    <option value="En attente" <?php echo ($filter_status === 'En attente') ? 'selected' : ''; ?>>En attente</option>
                    <option value="Partiellement livrée" <?php echo ($filter_status === 'Partiellement livrée') ? 'selected' : ''; ?>>Partiellement livrée</option>
                    <option value="Livrée" <?php echo ($filter_status === 'Livrée') ? 'selected' : ''; ?>>Livrée</option>
                </select>
            </div>

            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID Commande</th>
                            <th>Date Commande</th>
                            <th>Magasinier</th>
                            <th>Statut Commande</th>
                            <th>Statut Livraison</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($commandes)): ?>
                            <tr><td colspan="6" class="text-center">Aucune commande trouvée.</td></tr>
                        <?php else: ?>
                            <?php foreach ($commandes as $commande): ?>
                                <tr data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                    <td><?php echo htmlspecialchars($commande['id_commande_pneu']); ?></td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($commande['date_commande'])); ?></td>
                                    <td><?php echo htmlspecialchars($commande['nom_magasinier']); ?></td>
                                    <td><span class="status-badge status-<?php echo str_replace(' ', '-', htmlspecialchars($commande['statut_commande'])); ?>"><?php echo htmlspecialchars($commande['statut_commande']); ?></span></td>
                                    <td><span class="status-badge status-<?php echo str_replace(' ', '-', htmlspecialchars($commande['statut_livraison'])); ?>"><?php echo htmlspecialchars($commande['statut_livraison']); ?></span></td>
                                    <td class="table-actions">
                                        <button type="button" class="btn btn-info btn-sm toggle-details-btn" data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>" title="Voir Détails">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <?php if ($commande['statut_livraison'] === 'Partiellement livrée'): ?>
                                            <form action="commande_pneu.php" method="POST" style="display:inline;" onsubmit="return confirm('Êtes-vous sûr de vouloir clôturer cette livraison ?');">
                                                <input type="hidden" name="action" value="cloturer_livraison">
                                                <input type="hidden" name="id_commande_pneu" value="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                                <button type="submit" class="btn btn-success btn-sm">
                                                    <i class="fas fa-check-circle"></i> Clôturer
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                        <button type="button" class="btn btn-primary btn-sm generate-bon-btn" data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>" title="Générer Bon de Commande">
                                            <i class="fas fa-file-pdf"></i>
                                        </button>
                                    </td>
                                </tr>
                                <tr class="details-row" >
                                    <td colspan="6">
                                        <div class="details-row-content" id="details-content-<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                            <h4>Détails de la commande N°<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>:</h4>
                                            <?php if (!empty($commande['commentaire'])): ?>
                                                <p><strong>Commentaire :</strong></p>
                                                <div class="commentaire-details"><?php echo htmlspecialchars($commande['commentaire']); ?></div>
                                            <?php endif; ?>
                                            <div class="table-responsive">
                                                <table>
                                                    <thead>
                                                        <tr>
                                                            <th>Modèle Pneu</th>
                                                            <th>Marque</th>
                                                            <th>Dimension</th>
                                                            <th>Profil</th>
                                                            <th>Quantité Commandée</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody id="details-table-body-<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                                        </tbody>
                                                </table>
                                            </div>
                                            <?php if ($commande['statut_livraison'] === 'En attente'): ?>
                                                <div class="flex justify-end mt-4">
                                                    <button type="button" class="btn btn-warning btn-sm edit-commande-btn" data-commande-id="<?php echo htmlspecialchars($commande['id_commande_pneu']); ?>">
                                                        <i class="fas fa-edit"></i> Modifier cette commande
                                                    </button>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <div class="pagination">
                <?php if ($current_page_commandes > 1): ?>
                    <a href="?page_commandes=<?php echo $current_page_commandes - 1; ?><?php echo !empty($filter_status) ? '&filter_status=' . htmlspecialchars($filter_status) : ''; ?>">Précédent</a>
                <?php else: ?>
                    <span class="disabled">Précédent</span>
                <?php endif; ?>
                <?php for ($i = 1; $i <= $total_pages_commandes; $i++): ?>
                    <?php if ($i == $current_page_commandes): ?>
                        <span class="current-page"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="?page_commandes=<?php echo $i; ?><?php echo !empty($filter_status) ? '&filter_status=' . htmlspecialchars($filter_status) : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                <?php if ($current_page_commandes < $total_pages_commandes): ?>
                    <a href="?page_commandes=<?php echo $current_page_commandes + 1; ?><?php echo !empty($filter_status) ? '&filter_status=' . htmlspecialchars($filter_status) : ''; ?>">Suivant</a>
                <?php else: ?>
                    <span class="disabled">Suivant</span>
                <?php endif; ?>
            </div>
        </div>

    </div>

    <script>
        // Fonction globale pour afficher les messages
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            const navLinks = document.querySelectorAll('.navbar-nav a');

            // Gère l'ouverture/fermeture du menu burger
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Gère l'affichage du menu déroulant (dropdown) au CLIC
            const dropdownToggle = document.querySelector('.dropdown-toggle');
            if (dropdownToggle) {
                dropdownToggle.addEventListener('click', function(e) {
                    e.preventDefault(); // Empêche la navigation vers '#'
                    this.closest('.dropdown').classList.toggle('active');
                    // Fermer les autres dropdowns (si plusieurs, ce qui n'est pas le cas ici mais bonne pratique)
                    document.querySelectorAll('.dropdown.active').forEach(otherDropdown => {
                        if (otherDropdown !== this.closest('.dropdown')) {
                            otherDropdown.classList.remove('active');
                        }
                    });
                });
            }

            // Fermer le dropdown si l'on clique en dehors
            document.addEventListener('click', function(event) {
                const dropdown = document.querySelector('.navbar-nav .dropdown');
                // S'assurer que le clic n'est pas à l'intérieur du dropdown ET que ce n'est pas le toggle lui-même
                if (dropdown && !dropdown.contains(event.target) && event.target !== dropdownToggle) {
                    dropdown.classList.remove('active');
                    // Si le menu burger est ouvert et que c'est un clic à l'extérieur du navbar-nav sur mobile, le fermer aussi
                    if (navbarNav.classList.contains('open') && window.innerWidth <= 768) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                    }
                }
            });
            
            // --- Gestion du lien actif dans la navigation ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop();

                navLinks.forEach(link => {
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    if (linkFilename === currentFilename) {
                        link.classList.add('active');
                        const parentDropdown = link.closest('.dropdown');
                        if (parentDropdown) {
                            parentDropdown.classList.add('active');
                        }
                    } else {
                        link.classList.remove('active');
                    }
                });
            }
            setActiveLink();

            // --- Logique du formulaire de commande de pneus ---
            const commandeForm = document.getElementById('commandeForm');
            const commandeLinesContainer = document.getElementById('commande_lines_container');
            const addLineBtn = document.getElementById('add_line_btn');
            const submitCommandeBtn = document.getElementById('submit_commande_btn');
            const cancelEditCommandeBtn = document.getElementById('cancel_edit_commande_btn');
            // L'input caché pour l'ID de commande utilise maintenant le nom de colonne correct
            const idCommandeInput = document.getElementById('id_commande_input'); // L'ID JS reste le même pour la facilité
            const commandLinesInput = document.getElementById('command_lines_input');
            const commentaireInput = document.getElementById('commentaire');

            let lineCounter = 0; // Pour générer des IDs uniques pour chaque ligne

            const allPneusModels = <?php echo json_encode($all_pneus_models); ?>; // Données des modèles de pneus depuis PHP

            // Fonction pour générer les options du select, avec un filtre
            function generatePneuOptions(filterText = '') {
                let optionsHtml = '<option value="">Sélectionnez un modèle de pneu</option>';
                const lowerFilter = filterText.toLowerCase();

                allPneusModels.filter(pneu => 
                    pneu.nom_modele.toLowerCase().includes(lowerFilter) ||
                    pneu.nom_marque.toLowerCase().includes(lowerFilter) ||
                    pneu.valeur_dimension.toLowerCase().includes(lowerFilter) ||
                    pneu.nom_profil.toLowerCase().includes(lowerFilter)
                ).forEach(pneu => {
                    optionsHtml += `<option value="${pneu.id_pneu}">${pneu.nom_modele} (${pneu.nom_marque} - ${pneu.valeur_dimension} - ${pneu.nom_profil})</option>`;
                });
                return optionsHtml;
            }

            function addCommandeLine(pneuId = '', quantite = '') {
                const newLine = document.createElement('div');
                newLine.classList.add('commande-line-item');
                newLine.dataset.lineId = lineCounter; // Pour identifier la ligne
                
                // On génère toutes les options pour la première fois
                let optionsHtml = generatePneuOptions();
                
                newLine.innerHTML = `
                    <div class="form-group flex-1">
                        <label for="pneu_filter_${lineCounter}">Filtrer le pneu :</label>
                        <input type="text" id="pneu_filter_${lineCounter}" class="w-full" placeholder="Nom, marque, dimension...">
                    </div>
                    <div class="form-group flex-1">
                        <label for="pneu_model_${lineCounter}">Modèle de Pneu :</label>
                        <select name="pneu_model[${lineCounter}][id_pneu]" id="pneu_model_${lineCounter}" required>
                            ${optionsHtml}
                        </select>
                    </div>
                    <div class="form-group flex-1">
                        <label for="quantite_${lineCounter}">Quantité Commandée :</label>
                        <input type="number" name="pneu_model[${lineCounter}][quantite_commandee]" id="quantite_${lineCounter}" min="1" value="${quantite}" required>
                    </div>
                    <button type="button" class="remove-line-btn" data-line-id="${lineCounter}"><i class="fas fa-times"></i></button>
                `;
                
                commandeLinesContainer.appendChild(newLine);
                
                // Mettre à jour l'option sélectionnée si un pneuId est fourni (pour la modification)
                if (pneuId) {
                    const pneuSelect = newLine.querySelector('select');
                    pneuSelect.value = pneuId;
                }

                // Ajouter l'écouteur d'événements pour le bouton de suppression
                newLine.querySelector('.remove-line-btn').addEventListener('click', function() {
                    newLine.remove();
                    updateCommandLinesInput(); // Mettre à jour l'input caché après suppression
                });
                
                // Écouteur pour le filtrage en temps réel sur la ligne
                const pneuFilterInput = newLine.querySelector(`#pneu_filter_${lineCounter}`);
                const pneuSelect = newLine.querySelector(`#pneu_model_${lineCounter}`);
                pneuFilterInput.addEventListener('input', function() {
                    const filterValue = this.value;
                    const optionsHtml = generatePneuOptions(filterValue);
                    const selectedPneuId = pneuSelect.value;
                    pneuSelect.innerHTML = optionsHtml;
                    if (selectedPneuId && pneuSelect.querySelector(`option[value="${selectedPneuId}"]`)) {
                        pneuSelect.value = selectedPneuId;
                    }
                });

                lineCounter++;
            }

            // Fonction pour mettre à jour l'input caché `command_lines_input`
            function updateCommandLinesInput() {
                const lines = [];
                document.querySelectorAll('.commande-line-item').forEach(lineElement => {
                    const pneuId = lineElement.querySelector('select[name*="id_pneu"]').value;
                    const quantite = lineElement.querySelector('input[name*="quantite_commandee"]').value;
                    if (pneuId && quantite && parseInt(quantite) > 0) {
                        lines.push({
                            id_pneu: pneuId,
                            quantite_commandee: quantite
                        });
                    }
                });
                commandLinesInput.value = JSON.stringify(lines);
            }

            // Ajouter une ligne au chargement de la page si aucune n'existe (pour la création)
            if (commandeLinesContainer.children.length === 0) {
                addCommandeLine();
            }

            // Événement pour ajouter une nouvelle ligne
            addLineBtn.addEventListener('click', () => addCommandeLine());

            // Soumission du formulaire
            commandeForm.addEventListener('submit', function(e) {
                // Mettre à jour l'input caché avant l'envoi du formulaire
                updateCommandLinesInput();
                // Si aucune ligne n'est ajoutée, empêcher la soumission
                const lines = JSON.parse(commandLinesInput.value);
                if (lines.length === 0) {
                    e.preventDefault();
                    showMessage("Veuillez ajouter au moins une ligne à la commande.", "error");
                }
            });

            // --- Logique de modification de commande ---
            // Le gestionnaire d'événements pour les boutons d'édition est maintenant attaché aux boutons dans les détails
            // et déléguerait ou serait réattaché après le chargement des détails.
            // Pour simplifier, nous allons attacher le gestionnaire d'événements à un parent commun
            // qui est toujours présent dans le DOM, et utiliser la délégation d'événements.
            document.querySelector('.main-content').addEventListener('click', async function(event) {
                if (event.target.closest('.edit-commande-btn')) {
                    const button = event.target.closest('.edit-commande-btn');
                    const commandeId = button.dataset.commandeId;
                    try {
                        const response = await fetch(`commande_pneu.php?action=get_commande_details&id_commande_pneu=${commandeId}`);
                        const data = await response.json();

                        if (data.success) {
                            // Nettoyer les lignes existantes du formulaire
                            commandeLinesContainer.innerHTML = '';
                            lineCounter = 0; // Réinitialiser le compteur de lignes

                            // Remplir le formulaire principal (id_commande_input correspond à id_commande_pneu)
                            idCommandeInput.value = data.commande.id_commande_pneu;
                            commentaireInput.value = data.commande.commentaire || '';
                            submitCommandeBtn.innerHTML = '<i class="fas fa-save"></i> Enregistrer Modifications';
                            submitCommandeBtn.classList.remove('btn-success');
                            submitCommandeBtn.classList.add('btn-primary');
                            cancelEditCommandeBtn.style.display = 'inline-block';
                            
                            // Remplir les lignes de commande
                            data.details.forEach(detail => {
                                addCommandeLine(detail.id_pneu, detail.quantite_commandee);
                            });

                            showMessage("Commande N°" + commandeId + " chargée pour modification.", "info");
                            window.scrollTo({ top: commandeForm.offsetTop, behavior: 'smooth' }); // Remonter vers le formulaire
                        } else {
                            showMessage("Erreur lors du chargement de la commande : " + data.message, "error");
                        }
                    } catch (error) {
                        console.error('Erreur AJAX lors du chargement de la commande:', error);
                        showMessage("Erreur technique lors du chargement de la commande.", "error");
                    }
                }
            });


            // Annuler la modification
            cancelEditCommandeBtn.addEventListener('click', function() {
                idCommandeInput.value = ''; // Réinitialiser l'ID de commande (qui est id_commande_pneu)
                commandeLinesContainer.innerHTML = ''; // Vider les lignes
                commentaireInput.value = ''; // Vider le commentaire
                lineCounter = 0; // Réinitialiser le compteur
                addCommandeLine(); // Ajouter une ligne vide pour la création
                submitCommandeBtn.innerHTML = '<i class="fas fa-file-circle-plus"></i> Créer la commande';
                submitCommandeBtn.classList.remove('btn-primary');
                submitCommandeBtn.classList.add('btn-success');
                this.style.display = 'none'; // Cacher le bouton d'annulation
                showMessage("Modification annulée. Prêt pour une nouvelle commande.", "info");
            });

            // --- Logique d'affichage des détails de commande (avec l'icône œil) ---
            document.querySelectorAll('.toggle-details-btn').forEach(button => {
                button.addEventListener('click', async function() {
                    const commandeId = this.dataset.commandeId;
                    // Utilise l'ID direct pour cibler le contenu des détails
                    const detailsRowContent = document.getElementById(`details-content-${commandeId}`); 
                    const detailsTableBody = document.getElementById(`details-table-body-${commandeId}`);
                    const icon = this.querySelector('i');

                    if (!detailsRowContent.classList.contains('active')) { // Vérifier si n'est PAS actif
                        // Si les détails sont cachés, les charger et les afficher
                        try {
                            const response = await fetch(`commande_pneu.php?action=get_commande_details&id_commande_pneu=${commandeId}`);
                            const data = await response.json();

                            if (data.success && data.details) {
                                detailsTableBody.innerHTML = ''; // Nettoyer avant d'ajouter
                                data.details.forEach(detail => {
                                    const row = document.createElement('tr');
                                    row.innerHTML = `
                                        <td>${detail.nom_modele}</td>
                                        <td>${detail.nom_marque}</td>
                                        <td>${detail.valeur_dimension}</td>
                                        <td>${detail.nom_profil}</td>
                                        <td>${detail.quantite_commandee}</td>
                                    `;
                                    detailsTableBody.appendChild(row);
                                });
                                detailsRowContent.classList.add('active'); // Ajoute la classe 'active' pour dérouler
                                icon.classList.replace('fa-eye', 'fa-eye-slash'); // Changer l'icône
                                this.title = 'Masquer Détails'; // Mettre à jour le titre
                            } else {
                                showMessage("Erreur lors du chargement des détails : " + data.message, "error");
                            }
                        } catch (error) {
                            console.error('Erreur AJAX lors du chargement des détails:', error);
                            showMessage("Erreur technique lors du chargement des détails.", "error");
                        }
                    } else {
                        // Si les détails sont visibles, les masquer
                        detailsRowContent.classList.remove('active'); // Retire la classe 'active' pour enrouler
                        icon.classList.replace('fa-eye-slash', 'fa-eye'); // Changer l'icône
                        this.title = 'Voir Détails'; // Mettre à jour le titre
                    }
                });
            });

            // --- Logique de filtrage ---
            document.getElementById('filter_status').addEventListener('change', function() {
                const selectedStatus = this.value;
                let url = new URL(window.location.href);
                if (selectedStatus === 'all') {
                    url.searchParams.delete('filter_status');
                } else {
                    url.searchParams.set('filter_status', selectedStatus);
                }
                url.searchParams.delete('page_commandes'); // Réinitialiser la pagination
                window.location.href = url.toString();
            });

            // --- Génération de Bon de Commande (PDF) ---
            document.querySelectorAll('.generate-bon-btn').forEach(button => {
                button.addEventListener('click', function() {
                    const commandeId = this.dataset.commandeId; // Ceci correspond à id_commande_pneu
                    // Ouvrir une nouvelle fenêtre/onglet pour le PDF.
                    // Assurez-vous que generate_bon_pdf.php existe et gère la génération.
                    // Ce fichier devra récupérer les données de la commande, générer le PDF avec les bonnes infos
                    // et masquer les prix au magasinier.
                    window.open(`php/generate_bon_commande_pneu_pdf.php?id_commande_pneu=${commandeId}`, '_blank');
                });
            });
            
        });
        
    </script>
</body>
</html>