<?php
// ASE_Maintenance/commande.php

session_start(); // Toujours commencer par démarrer la session

// Pour le débogage : afficher toutes les erreurs PHP directement dans la sortie
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';
// Vérification de l'authentification et du rôle Magasinier
if (
    empty($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Magasinier'
) {
    header('Location: index'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

$message = '';
$message_type = '';

// Récupérer l'ID du magasinier connecté depuis la session
$id_magasinier_connecte = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;

// Vérifiez si l'utilisateur est bien connecté avant de continuer
if ($id_magasinier_connecte === null) {
    // Redirigez vers la page de connexion (index.php) ou affichez un message d'erreur
    $_SESSION['message'] = "Vous devez être connecté pour accéder à cette page.";
    $_SESSION['message_type'] = 'error';
    header("Location: index.php"); // Redirection vers index.php
    exit();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Déterminer s'il s'agit d'une modification ou d'une création en fonction des *valeurs* des champs cachés
    $is_modification = (isset($_POST['modifier_commande']) && $_POST['modifier_commande'] === '1');
    $is_creation = (isset($_POST['creer_commande']) && $_POST['creer_commande'] === '1');

    if ($is_creation || $is_modification) { // Procéder uniquement s'il s'agit clairement de l'un ou l'autre
        $id_commande = null; // Initialiser $id_commande
        // Récupérer le commentaire du formulaire
        $commentaire = isset($_POST['commentaire']) ? trim($_POST['commentaire']) : null;
        if (empty($commentaire)) {
            $commentaire = null; // S'assurer que la valeur est NULL si le champ est vide
        }

        try {
            $pdo->beginTransaction();

            if ($is_modification) {
                $id_commande = filter_var($_POST['modified_commande_id'], FILTER_VALIDATE_INT);
                if (!$id_commande) {
                    throw new Exception("ID de commande invalide pour la modification.");
                }
                // Supprimer les détails de commande existants
                $stmt_delete_details = $pdo->prepare("DELETE FROM details_commande_appro WHERE id_commande = ?");
                $stmt_delete_details->execute([$id_commande]);

                // Mettre à jour la commande principale, incluant le commentaire
                $stmt_update_commande_main = $pdo->prepare("UPDATE commandes_appro SET date_commande = NOW(), id_magasinier = ?, statut_commande = 'En attente', statut_livraison = 'En attente', commentaire = ? WHERE id_commande = ?");
                $stmt_update_commande_main->execute([$id_magasinier_connecte, $commentaire, $id_commande]);


            } else { // Il s'agit d'une création
                // Créer une nouvelle commande, incluant le commentaire
                $stmt = $pdo->prepare("INSERT INTO commandes_appro (id_magasinier, commentaire) VALUES (?, ?)");
                $stmt->execute([$id_magasinier_connecte, $commentaire]);
                $id_commande = $pdo->lastInsertId();
            }

            if (isset($_POST['pieces']) && is_array($_POST['pieces'])) {
                // CORRECTION: Ajout de 'quantite_recue' à la requête d'insertion
                $stmt_detail = $pdo->prepare(
                    "INSERT INTO details_commande_appro (id_commande, id_piece, quantite_commandee, prix_ht_unitaire, tva_unitaire, quantite_recue)
                    VALUES (?, ?, ?, ?, ?, ?)"
                );

                foreach ($_POST['pieces'] as $piece_data) {
                    $id_piece = filter_var($piece_data['id_piece'], FILTER_VALIDATE_INT);
                    $quantite_commandee = filter_var($piece_data['quantite_commandee'], FILTER_VALIDATE_FLOAT);
                    $prix_ht_unitaire = filter_var($piece_data['prix_ht_unitaire'], FILTER_VALIDATE_FLOAT);
                    $tva_unitaire = filter_var($piece_data['tva_unitaire'], FILTER_VALIDATE_FLOAT);
                    // Initialiser quantite_recue à 0 pour les nouvelles commandes ou modifications
                    $quantite_recue = 0; 

                    // Assurez-vous que les données sont valides avant l'insertion
                    if ($id_piece && $quantite_commandee > 0 && $prix_ht_unitaire >= 0 && $tva_unitaire >= 0) {
                        // CORRECTION: Ajout de $quantite_recue à l'exécution de la requête
                        $stmt_detail->execute([$id_commande, $id_piece, $quantite_commandee, $prix_ht_unitaire, $tva_unitaire, $quantite_recue]);
                    }
                }
            }

            $pdo->commit();
            // Stocker le message de succès dans la session
            $_SESSION['message'] = ($is_modification ? "Commande n° " . htmlspecialchars($id_commande) . " modifiée avec succès !" : "Commande n° " . htmlspecialchars($id_commande) . " créée avec succès !");
            $_SESSION['message_type'] = 'success';

        } catch (Exception $e) { // Changed to Exception to catch custom exceptions
            $pdo->rollBack();
            // Stocker le message d'erreur dans la session
            $_SESSION['message'] = "Erreur lors de la " . ($is_modification ? "modification" : "création") . " de la commande : " . $e->getMessage();
            $_SESSION['message_type'] = 'error';
        }

        // Redirection après POST pour éviter le message "resend data"
        header("Location: commande.php");
        exit();
    }
}

// Récupérer le message de la session s'il existe et le supprimer
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']); // Supprimer le message après l'avoir affiché
    unset($_SESSION['message_type']);
}

// Récupérer toutes les pièces disponibles
$pieces_disponibles = [];
try {
    $stmt = $pdo->query("SELECT id_piece, nom_piece, code_piece, reference, prix_unitaire, tva FROM piece ORDER BY nom_piece");
    $pieces_disponibles = $stmt->fetchAll();
} catch (PDOException $e) {
    $message = "Erreur lors du chargement des pièces : " . $e->getMessage();
    $message_type = 'error';
}

// --- Logique de filtrage et de pagination ---
$filter_statut_livraison = isset($_GET['statut_livraison']) ? $_GET['statut_livraison'] : '';
$records_per_page = 5;
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($current_page - 1) * $records_per_page;

$sql_base = "
    SELECT
        c.id_commande,
        c.date_commande,
        c.statut_commande,
        c.statut_livraison,
        c.commentaire,
        u.nom_utilisateur AS magasinier_nom
    FROM
        commandes_appro c
    JOIN
        utilisateurs u ON c.id_magasinier = u.id_utilisateur
";

$sql_count = "SELECT COUNT(*) FROM commandes_appro c"; // Ajout de l'alias c pour le WHERE

$where_clauses = [];
$params_filter = []; // Paramètres pour le filtre (pour le COUNT et la requête principale)

if (!empty($filter_statut_livraison)) {
    $where_clauses[] = "c.statut_livraison = ?";
    $params_filter[] = $filter_statut_livraison;
}

if (count($where_clauses) > 0) {
    $sql_base .= " WHERE " . implode(" AND ", $where_clauses);
    $sql_count .= " WHERE " . implode(" AND ", $where_clauses);
}

$sql_base .= " ORDER BY c.date_commande DESC LIMIT ? OFFSET ?";
$params_query = array_merge($params_filter, [$records_per_page, $offset]); // Tous les paramètres pour la requête principale

// Récupérer le nombre total de commandes pour la pagination
$stmt_count = $pdo->prepare($sql_count);
$stmt_count->execute($params_filter); // Exécuter avec SEULEMENT les paramètres de filtre
$total_records = $stmt_count->fetchColumn();
$total_pages = ceil($total_records / $records_per_page);


// Récupérer toutes les commandes avec leurs détails de pièces et le nom du magasinier
$commandes_avec_details = [];
try {
    $stmt_commandes = $pdo->prepare($sql_base);
    $stmt_commandes->execute($params_query); // Exécuter avec TOUS les paramètres
    $commandes_principales = $stmt_commandes->fetchAll();

    foreach ($commandes_principales as $commande) {
        // CORRECTION: Ajout de quantite_recue dans la sélection des détails
        $stmt_details = $pdo->prepare("
            SELECT
                dc.quantite_commandee,
                dc.prix_ht_unitaire,
                dc.tva_unitaire,
                dc.quantite_recue, -- Ajout de quantite_recue ici
                p.id_piece,
                p.nom_piece,
                p.reference,
                p.code_piece
            FROM
                details_commande_appro dc
            JOIN
                piece p ON dc.id_piece = p.id_piece
            WHERE
                dc.id_commande = ?
        ");
        $stmt_details->execute([$commande['id_commande']]);
        $commande['details'] = $stmt_details->fetchAll();

        // Calculer le total TTC pour cette commande
        $total_ttc_commande = 0;
        foreach ($commande['details'] as $detail) {
            $prix_ht = (float)$detail['prix_ht_unitaire'];
            $tva_pourcentage = (float)$detail['tva_unitaire'];
            $quantite = (float)$detail['quantite_commandee'];
            $prix_ttc_unitaire = $prix_ht * (1 + ($tva_pourcentage / 100));
            $total_ligne_ttc = $prix_ttc_unitaire * $quantite;
            $total_ttc_commande += $total_ligne_ttc;
        }
        $commande['total_ttc'] = $total_ttc_commande;

        $commandes_avec_details[] = $commande;
    }

} catch (PDOException $e) {
    $message = "Erreur lors du chargement des commandes avec détails : " . $e->getMessage();
    $message_type = 'error';
}

// Statuts de livraison possibles pour le filtre (vous pouvez les récupérer dynamiquement de la base si nécessaire)
$statuts_livraison = ['En attente', 'Partiellement livrée', 'Livrée'];

// AJAX handler for 'cloturer_commande' and 'get_order_details_for_edit'
if (isset($_GET['action'])) {
    header('Content-Type: application/json'); // S'assurer que la réponse est au format JSON

    switch ($_GET['action']) {
        case 'cloturer_commande':
            if (isset($_POST['id_commande'])) {
                $id_commande = filter_var($_POST['id_commande'], FILTER_VALIDATE_INT);
                if ($id_commande) {
                    try {
                        $pdo->beginTransaction();
                        // error_log("[Cloture Debug] Début de la clôture pour commande ID: " . $id_commande);

                        // 1. Vérifier le statut actuel et s'assurer qu'il est 'Partiellement livrée'
                        $stmt_check_status = $pdo->prepare("SELECT statut_livraison FROM commandes_appro WHERE id_commande = ?");
                        $stmt_check_status->execute([$id_commande]);
                        $current_status = $stmt_check_status->fetchColumn();

                        if ($current_status === false) {
                            // error_log("[Cloture Debug] Commande introuvable pour ID: " . $id_commande);
                            throw new Exception('Commande introuvable.');
                        }
                        if ($current_status !== 'Partiellement livrée') {
                            // error_log("[Cloture Debug] Tentative de clôture d'une commande avec statut incorrect: " . $current_status . " pour ID: " . $id_commande);
                            throw new Exception('La commande ne peut être clôturée car son statut est "' . htmlspecialchars($current_status) . '". Seules les commandes "Partiellement livrée" peuvent l\'être.');
                        }
                        // error_log("[Cloture Debug] Statut actuel ('Partiellement livrée') vérifié pour ID: " . $id_commande);

                        // 2. Parcourir les détails de commande pour cette commande
                        $stmt_details_commande = $pdo->prepare("SELECT id_piece, quantite_commandee, prix_ht_unitaire, tva_unitaire FROM details_commande_appro WHERE id_commande = ?");
                        $stmt_details_commande->execute([$id_commande]);
                        $details_commande = $stmt_details_commande->fetchAll(PDO::FETCH_ASSOC);

                        if (empty($details_commande)) {
                             // error_log("[Cloture Debug] Aucun détail de pièce trouvé pour commande ID: " . $id_commande);
                             throw new Exception('Aucun détail de pièce trouvé pour cette commande.');
                        }

                        // Suppression des variables de totaux, car non utilisées pour la mise à jour principale
                        // $new_total_ht_commande = 0;
                        // $new_total_tva_commande = 0;
                        // $new_total_ttc_commande = 0;

                        foreach ($details_commande as $detail) {
                            $id_piece = $detail['id_piece'];
                            // $prix_ht_unitaire = (float)$detail['prix_ht_unitaire']; // Non utilisé directement pour la MAJ du statut/quantité
                            // $tva_unitaire = (float)$detail['tva_unitaire']; // Non utilisé directement pour la MAJ du statut/quantité
                            // error_log("[Cloture Debug] Traitement pièce ID: " . $id_piece . " pour commande ID: " . $id_commande);

                            // CORRECTION ICI: Changer la table et la colonne
                            // Récupérer la quantité totale reçue pour cette pièce dans cette commande
                            $stmt_total_recue = $pdo->prepare("
                                SELECT COALESCE(SUM(lrc.quantite_receptionnee), 0) AS total_recue
                                FROM lignes_reception lrc
                                JOIN receptions r ON lrc.id_reception = r.id_reception
                                WHERE r.id_commande = ? AND lrc.id_piece = ?
                            ");
                            $stmt_total_recue->execute([$id_commande, $id_piece]);
                            $total_recue_for_piece = (int)$stmt_total_recue->fetchColumn();
                            // error_log("[Cloture Debug] Pièce ID: " . $id_piece . ", Quantité reçue: " . $total_recue_for_piece);


                            // Mettre à jour quantite_commandee dans details_commande_appro pour être égale à quantite_recue
                            $stmt_update_detail_qty = $pdo->prepare("UPDATE details_commande_appro SET quantite_commandee = ? WHERE id_commande = ? AND id_piece = ?");
                            $stmt_update_detail_qty->execute([$total_recue_for_piece, $id_commande, $id_piece]);
                            // error_log("[Cloture Debug] Quantité commandée mise à jour pour pièce ID: " . $id_piece . ", Lignes affectées: " . $stmt_update_detail_qty->rowCount());


                            // Recalculer les totaux de ligne basés sur la nouvelle quantite_commandee
                            // Ces totaux ne sont plus utilisés pour la mise à jour de la table `commandes_appro`
                            // $recalculated_prix_total_ht_ligne = $total_recue_for_piece * $prix_ht_unitaire;
                            // $recalculated_tva_ligne = $recalculated_prix_total_ht_ligne * ($tva_unitaire / 100);
                            // $recalculated_prix_total_ttc_ligne = $recalculated_prix_total_ht_ligne + $recalculated_tva_ligne;
                            
                            // error_log("[Cloture Debug] Pièce ID: " . $id_piece . ", HT Recalculé: " . $recalculated_prix_total_ht_ligne . ", TVA Recalculée: " . $recalculated_tva_ligne . ", TTC Recalculé: " . $recalculated_prix_total_ttc_ligne);

                            // Sommer pour les nouveaux totaux principaux de la commande
                            // Ces sommes ne sont plus utilisées pour la mise à jour de la table `commandes_appro`
                            // $new_total_ht_commande += $recalculated_prix_total_ht_ligne;
                            // $new_total_tva_commande += $recalled_tva_ligne;
                            // $new_total_ttc_commande += $recalculated_prix_total_ttc_ligne;
                        }

                        // 3. Mettre à jour SEULEMENT les statuts principaux de la commande
                        // Suppression de date_livraison_prevue, total_ht, total_tva, total_ttc
                        $stmt_update_commande_main = $pdo->prepare("
                            UPDATE commandes_appro
                            SET
                                statut_livraison = 'Livrée',
                                statut_commande = 'Terminee'
                            WHERE id_commande = ?
                        ");
                        $stmt_update_commande_main->execute([
                            $id_commande
                        ]);

                        $rows_updated = $stmt_update_commande_main->rowCount();
                        // error_log("[Cloture Debug] La commande principale n'a pas été mise à jour. Cela peut indiquer que la commande n'était pas 'Partiellement livrée' ou que l'ID n'existait pas (déjà vérifié).");
                        if ($rows_updated === 0) {
                            // Ceci ne devrait pas arriver si le check_status a réussi et qu'il y a des détails
                            throw new Exception('La commande principale n\'a pas pu être mise à jour.');
                        }


                        $pdo->commit();
                        // error_log("[Cloture Debug] Transaction commise avec succès pour commande ID: " . $id_commande);
                        echo json_encode(['success' => true, 'message' => 'Commande clôturée et mise à jour avec succès.']);

                    } catch (Exception $e) {
                        $pdo->rollBack();
                        error_log("[Cloture Debug] Erreur lors de la clôture (ROLLBACK) pour commande ID: " . $id_commande . " - Erreur: " . $e->getMessage());
                        echo json_encode(['success' => false, 'message' => 'Erreur lors de la clôture de la commande : ' . $e->getMessage()]);
                    }
                } else {
                    // error_log("[Cloture Debug] ID de commande invalide reçu.");
                    echo json_encode(['success' => false, 'message' => 'ID de commande invalide.']);
                }
            } else {
                // error_log("[Cloture Debug] ID de commande manquant dans la requête POST.");
                echo json_encode(['success' => false, 'message' => 'ID de commande manquant.']);
            }
            break;

        case 'get_order_details_for_edit':
            if (isset($_GET['id_commande'])) {
                $id_commande = filter_var($_GET['id_commande'], FILTER_VALIDATE_INT);
                if ($id_commande) {
                    try {
                        // Inclure le commentaire dans la sélection
                        $stmt_order = $pdo->prepare("SELECT id_commande, statut_livraison, commentaire FROM commandes_appro WHERE id_commande = ?");
                        $stmt_order->execute([$id_commande]);
                        $order_info = $stmt_order->fetch(PDO::FETCH_ASSOC);

                        if (!$order_info) {
                            echo json_encode(['success' => false, 'message' => 'Commande introuvable.']);
                            exit();
                        }
                        
                        if ($order_info['statut_livraison'] !== 'En attente') {
                            echo json_encode(['success' => false, 'message' => 'Seules les commandes "En attente" peuvent être modifiées.']);
                            exit();
                        }

                        // CORRECTION: Ajout de quantite_recue dans la sélection des détails pour l'édition
                        $stmt_details_for_edit = $pdo->prepare("
                            SELECT
                                dc.id_piece,
                                dc.quantite_commandee,
                                dc.prix_ht_unitaire,
                                dc.tva_unitaire,
                                dc.quantite_recue, -- Ajout de quantite_recue ici
                                p.nom_piece,
                                p.reference,
                                p.code_piece
                            FROM
                                details_commande_appro dc
                            JOIN
                                piece p ON dc.id_piece = p.id_piece
                            WHERE
                                dc.id_commande = ?
                        ");
                        $stmt_details_for_edit->execute([$id_commande]);
                        $pieces_for_edit = $stmt_details_for_edit->fetchAll(PDO::FETCH_ASSOC);

                        // Renvoyer les détails de la commande et le commentaire
                        echo json_encode(['success' => true, 'commande_id' => $id_commande, 'commentaire' => $order_info['commentaire'], 'pieces' => $pieces_for_edit]);

                    } catch (PDOException $e) {
                        error_log("Erreur PDO (get_order_details_for_edit) : " . $e->getMessage());
                        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors de la récupération des détails de commande pour modification.']);
                    }
                } else {
                    echo json_encode(['success' => false, 'message' => 'ID de commande invalide.']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'ID de commande manquant.']);
            }
            break;
        default:
            echo json_encode(['error' => 'Action AJAX non reconnue.']);
            break;
    }
    exit(); // Terminate script after AJAX request
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des Commandes d'Approvisionnement - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />

    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js"></script>
    <script src="https://unpkg.com/xlsx/dist/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>

    <script src="https://cdn.tailwindcss.com"></script>

    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --form-gap: 15px;
            --form-field-padding: 10px;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

    .navbar-brand span {
        font-size: 1em;
        font-weight: bold;
        color: white;
    }

    .navbar-nav {
        list-style: none;
        margin: 0;
        padding: 0;
        display: flex;
    }

    .navbar-nav li {
        margin-left: 30px;
    }

    .navbar-nav a {
        color: var(--text-color);
        text-decoration: none;
        padding: 8px 0;
        transition: color 0.3s ease, border-bottom 0.3s ease;
        position: relative;
        display: flex;
        align-items: center;
        font-size: 0.7em;
    }

    .navbar-nav a i {
        margin-right: 10px;
        font-size: 0.7em;
    }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px; /* Taille cohérente avec sortie.php */
            width: 95%; /* Taille cohérente avec sortie.php */
            margin: 30px auto; /* Marge cohérente */
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex; /* Aligner icône et texte */
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
            transition: opacity 0.3s ease-in-out; /* Pour l'effet de chargement AJAX */
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em;
        }

        .form-group input[type="number"],
        .form-group select,
        .form-group input[type="text"][readonly],
        .form-group .filter-piece-input,
        .form-group textarea { /* Added textarea to styling */
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-group input[type="text"][readonly] {
            background-color: rgba(var(--input-bg), 0.5); /* Plus clair pour les champs readonly */
            cursor: not-allowed;
            opacity: 0.8;
        }

        .form-group input[type="number"]:focus,
        .form-group select:focus,
        .form-group input[type="text"][readonly]:focus,
        .form-group .filter-piece-input:focus,
        .form-group textarea:focus { /* Added textarea to focus styling */
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3);
            outline: none;
        }

        /* Buttons styles */
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white;
            margin-top: 15px; /* Adjust as needed */
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 8px 15px; font-size: 0.9em; }
        /* Style pour les boutons avec juste une icône */
        .btn-icon {
            padding: 8px; /* Padding plus petit pour les icônes */
            width: 40px; /* Taille fixe pour les boutons icônes */
            height: 40px; /* Taille fixe pour les boutons icônes */
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i {
            margin-right: 0 !important; /* Supprimer la marge pour les icônes seules */
        }

        .piece-entry {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
            gap: var(--form-gap);
            border: 1px solid var(--input-border);
            padding: 15px;
            border-radius: 8px;
            background-color: var(--input-bg); /* Couleur de fond pour chaque entrée de pièce */
            flex-wrap: wrap;
        }

        .piece-entry .form-group {
            flex: 1;
            min-width: 180px; /* Minimum width for form group inside piece entry */
            margin-bottom: 0; /* Override default form-group margin-bottom */
        }

        .remove-piece-btn {
            background-color: var(--btn-danger-bg);
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 1.1em;
            transition: background-color 0.3s ease;
        }

        .remove-piece-btn:hover {
            background-color: var(--btn-danger-hover);
        }

        /* --- Table styles --- */
        .table-responsive {
            overflow-x: auto; /* Permet le défilement horizontal */
            -webkit-overflow-scrolling: touch;
            margin-top: 30px;
            margin-bottom: 30px;
            border-radius: 8px; /* Bordure arrondie du conteneur */
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color); /* Bordure du conteneur */
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg); /* Fond du tableau */
            min-width: 900px; /* Largeur minimale pour le tableau avant de défiler */
        }

        th, td {
            border: 1px solid var(--input-border); /* Bordures des cellules */
            padding: 5px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg); /* Fond des en-têtes */
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg); /* Couleur de fond pour les lignes paires */
        }

        tr:hover {
            background-color: #3a3a3a; /* Effet hover sur les lignes */
            transition: background-color 0.2s ease;
        }

        .table-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-start;
            align-items: center;
        }

        /* Nouveaux styles pour le déroulant des détails */
        .details-toggle-btn {
            background: none;
            border: none;
            color: var(--btn-info-bg); /* Couleur pour l'icône de détails */
            cursor: pointer;
            font-size: 1.4em; /* Taille de l'icône */
            padding: 5px;
            transition: color 0.2s ease, transform 0.2s ease;
        }
        .details-toggle-btn:hover {
            color: var(--btn-info-hover);
            transform: scale(1.1);
        }

        .order-details-row {
            display: none; /* Caché par défaut */
            overflow: scroll;
            max-height: auto;
            background-color: #2a2a2a; /* Couleur légèrement différente pour les détails */
            border-top: 1px dashed var(--border-color);
        }

        .order-details-container {
            max-height: 0;
            overflow: hidden;
            transition: max-height 0.3s ease-out;
            padding: 0; /* Padding contrôlé par .order-details-container.open */
            margin-top: 0;
        }

        .order-details-container.open {
            max-height: 1500px; /* Suffisamment grand pour le contenu */
            padding-top: 15px; /* Nouveau padding */
            padding-bottom: 15px;
        }

        .order-details {
            padding: 0 15px; /* Padding interne pour les détails */
        }

        .order-details h4 {
            color: var(--heading-color);
            font-size: 1.2em;
            margin-bottom: 10px;
        }

        .order-details ul {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .order-details li {
            background-color: var(--input-bg); /* Fond des listes de détails */
            padding: 10px;
            margin-bottom: 8px;
            border-radius: 4px;
            display: flex;
            flex-direction: column;
            align-items: flex-start;
            border: 1px solid var(--input-border);
            box-shadow: 0 1px 3px rgba(0,0,0,0.2);
        }

        .order-details li span {
            font-weight: normal;
            font-size: 0.9em;
            color: var(--text-color);
            margin-top: 2px;
            opacity: 0.8;
        }

        .order-details li strong {
            color: var(--accent-color); /* Couleur pour les noms de pièces dans les détails */
            font-weight: bold;
            font-size: 1em;
            margin-bottom: 5px;
        }

        .details-total-ttc {
            font-weight: bold;
            color: var(--success-text); /* Vert pour le total TTC */
            font-size: 1.3em;
            text-align: right;
            margin-top: 20px;
            padding-top: 15px;
            border-top: 1px dashed var(--border-color);
        }

        .generate-bon-btn {
            background-color: var(--btn-info-bg); /* Vert pour générer bon */
            color: white;
            padding: 10px 18px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 1em;
            transition: background-color 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            white-space: nowrap;
        }
        .generate-bon-btn:hover {
            background-color: var(--btn-info-hover);
        }

        /* --- Pagination styles (CORRIGÉ) --- */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 30px;
            gap: 10px;
            flex-wrap: wrap; 
        }

        .pagination a, .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 10px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease;
            font-weight: bold;
            font-size: 1.1em;
            margin-bottom: 5px;
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
        }

        .pagination span.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* NOUVEAU STYLE pour les ellipses */
        .pagination span.ellipsis {
            background-color: transparent;
            font-weight: bold;
            cursor: default;
        }

        /* Filter form styling */
        .filter-section {
            background-color: var(--form-bg);
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        .filter-section .form-group {
            margin-bottom: 0; /* Remove extra margin */
        }

        .filter-section select {
            width: auto;
            min-width: 200px;
        }

        .filter-section button {
            margin-top: 0;
            padding: 10px 20px;
            font-size: 1em;
        }

        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 {
                font-size: 2.5em;
            }
            h2.section-title {
                font-size: 2em;
            }
            /* Pas de changement pour .piece-entry, il est déjà flex-wrap */
            .piece-entry .form-group {
                flex-basis: 48%; /* Deux colonnes sur les écrans moyens */
                min-width: unset; /* Supprime le min-width par default */
            }
            .piece-entry .remove-piece-btn {
                width: 100%; /* Le bouton prend toute la largeur */
                margin-top: 10px;
            }
            .filter-section form {
                flex-direction: column;
                align-items: flex-start;
                gap: 15px;
            }
            .filter-section select, .filter-section button {
                width: 100%;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav {
                display: none;
            }
            .burger-menu {
                display: block;
            }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 {
                font-size: 2.2em;
                margin-bottom: 25px;
            }
            h2.section-title {
                font-size: 1.8em;
                margin-top: 30px;
            }
            th, td {
                padding: 10px;
                font-size: 0.95em;
            }
            table {
                min-width: unset;
            } /* Allow table to shrink */
            .order-details li {
                flex-direction: column;
                align-items: flex-start;
            }
            .table-actions {
                flex-direction: column; /* Empile les boutons sur mobile */
                align-items: flex-start;
            }
            .generate-bon-btn {
                width: 100%; /* Les boutons prennent toute la largeur */
                justify-content: center;
            }
            /* Pour le formulaire de création de commande sur très petits écrans */
            .piece-entry .form-group {
                flex-basis: 100%; /* Chaque champ prend toute la largeur */
            }
            .btn {
                width: 100%;
                justify-content: center;
                margin-bottom: 12px;
            }
            .pagination {
                gap: 5px;
            }
            .pagination a, .pagination span {
                padding: 8px 12px;
                font-size: 0.9em;
            }
        }

        @media (max-width: 480px) {
            .main-content {
                padding: 15px;
                margin: 10px auto;
            }
            h1 {
                font-size: 1.8em;
                margin-bottom: 20px;
            }
            h2.section-title {
                font-size: 1.5em;
            }
            .navbar {
                padding: 10px 15px;
            }
            .navbar-brand img {
                height: 35px;
            }
            .navbar-brand span {
                font-size: 1.2em;
            }
            .form-section, .list-section {
                padding: 20px;
            }
            .btn {
                font-size: 1em;
                padding: 10px 20px;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="magasinier"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="piece"><i class="fas fa-boxes"></i> Gestion pièce</a></li>
            <li><a href="commande"  class="active"><i class="fas fa-shopping-cart"></i> Commande pièce</a></li>
            <li><a href="reception"><i class="fas fa-truck-loading"></i> Réception pièce</a></li>
            <li><a href="sortie"><i class="fas fa-truck-ramp-box"></i> Sortie pièce</a></li>
            <li><a href="pneu"><i class="fas fa-car"></i> Gestion Pneumatique </a></li> 
            <li><a href="php/authentification.php?action=logout"> <i style="color: red; font-size: 1.5em;" class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1>Gestion des Commandes d'Approvisionnement</h1>

        <?php if ($message): ?>
            <div class="message <?php echo $message_type; ?>">
                <?php
                    if ($message_type === 'success') echo '<i class="fas fa-check-circle"></i>';
                    else if ($message_type === 'error') echo '<i class="fas fa-times-circle"></i>';
                    else if ($message_type === 'warning') echo '<i class="fas fa-exclamation-triangle"></i>';
                ?>
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <div class="form-section">
            <h2 id="formTitle" class="section-title">Créer une nouvelle commande</h2>
            <form id="commandeForm" action="commande.php" method="POST">
                <input type="hidden" name="creer_commande" id="actionTypeCreer" value="1">
                <input type="hidden" name="modifier_commande" id="actionTypeModifier" value="0">
                <input type="hidden" name="modified_commande_id" id="modifiedCommandeIdInput" value="">
                
                <div id="piecesContainer">
                    </div>
                
                <div class="form-group mb-6">
                    <label for="commentaire">Commentaire de la commande:</label>
                    <textarea name="commentaire" id="commentaire" rows="4" placeholder="Ajouter un commentaire pour cette commande..."></textarea>
                </div>

                <div class="flex justify-between items-center mt-6">
                    <button type="button" id="addNewPiece" class="btn btn-info btn-sm">
                        <i class="fas fa-plus"></i> Ajouter une Pièce
                    </button>
                    <div class="flex gap-4">
                        <button type="submit" id="submitCommandeButton" class="btn btn-primary">
                            <i class="fas fa-save"></i> Créer la commande
                        </button>
                        <button type="button" id="cancelEditButton" class="btn btn-warning" style="display: none;">
                            <i class="fas fa-times-circle"></i> Annuler la modification
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <div class="list-section">
            <h2 class="section-title">Commandes Existantes</h2>

            <div class="filter-section mb-6">
                <form action="commande.php" method="GET" class="flex flex-wrap items-center gap-4">
                    <div class="form-group flex-1 min-w-[200px]">
                        <label for="filter_statut_livraison" class="block text-gray-300 text-sm font-bold mb-2">Filtrer par statut de livraison:</label>
                        <select name="statut_livraison" id="filter_statut_livraison" class="block w-full px-3 py-2 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                            <option value="">Tous les statuts</option>
                            <?php foreach ($statuts_livraison as $statut): ?>
                                <option value="<?php echo htmlspecialchars($statut); ?>" <?php echo ($filter_statut_livraison === $statut) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($statut); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-filter"></i> Filtrer
                    </button>
                    <a href="commande.php" class="btn btn-info">
                        <i class="fas fa-sync-alt"></i> Réinitialiser
                    </a>
                </form>
            </div>

            <div class="table-responsive">
                <?php if (empty($commandes_avec_details)): ?>
                    <p class="text-center text-gray-400 py-4">Aucune commande trouvée pour le moment.</p>
                <?php else: ?>
                    <table>
                        <thead>
                            <tr>
                                <th>ID Commande</th>
                                <th>Date Commande</th>
                                <th>Magasinier</th>
                                <th>Commentaire</th> <th>Statut Commande</th>
                                <th>Statut Livraison</th>
                                <th>Total TTC</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($commandes_avec_details as $commande): ?>
                            <tr class="order-row">
                                <td><?php echo htmlspecialchars($commande['id_commande']); ?></td>
                                <td><?php echo htmlspecialchars((new DateTime($commande['date_commande']))->format('d/m/Y H:i')); ?></td>
                                <td><?php echo htmlspecialchars($commande['magasinier_nom']); ?></td>
                                <td><?php echo nl2br(htmlspecialchars($commande['commentaire'])); ?></td>
                                <td><?php echo htmlspecialchars($commande['statut_commande']); ?></td>
                                <td><?php echo htmlspecialchars($commande['statut_livraison']); ?></td>
                                <td><?php echo number_format($commande['total_ttc'], 2, ',', ' '); ?> Ar</td>
                                <td class="table-actions">
                                    <button class="details-toggle-btn" data-order-id="<?php echo $commande['id_commande']; ?>" aria-expanded="false" aria-controls="details-<?php echo $commande['id_commande']; ?>">
                                        <i class="fas fa-chevron-down"></i>
                                    </button>
                                    <?php if ($commande['statut_livraison'] === 'En attente'): ?>
                                    <button class="btn btn-sm btn-warning edit-order-btn" data-id="<?php echo $commande['id_commande']; ?>">
                                        <i class="fas fa-edit"></i> Modifier
                                    </button>
                                    <?php elseif ($commande['statut_livraison'] === 'Partiellement livrée'): ?>
                                    <button class="btn btn-sm btn-success cloturer-commande-btn" data-id="<?php echo $commande['id_commande']; ?>">
                                        <i class="fas fa-check-double"></i> Clôturer
                                    </button>
                                    <?php endif; ?>
                                    <button class="btn btn-sm btn-info generate-bon-btn" data-id="<?php echo $commande['id_commande']; ?>">
                                        <i class="fas fa-file-pdf"></i> Bon
                                    </button>
                                </td>
                            </tr>
                            <tr class="order-details-row">
                                <td colspan="8"> <div id="details-<?php echo $commande['id_commande']; ?>" class="order-details-container" role="region" aria-hidden="true">
                                        <div class="order-details">
                                            <h4>Détails des pièces commandées:</h4>
                                            <ul>
                                                <?php foreach ($commande['details'] as $detail): ?>
                                                    <li>
                                                        <strong>Pièce: <?php echo htmlspecialchars($detail['nom_piece']); ?></strong><br>
                                                        <span>Code: <?php echo htmlspecialchars($detail['code_piece']); ?></span><br>
                                                        <span>Référence: <?php echo htmlspecialchars($detail['reference']); ?></span><br>
                                                        <span>Quantité commandée: <?php echo htmlspecialchars($detail['quantite_commandee']); ?></span><br>
                                                        <span>Prix HT unitaire: <?php echo number_format($detail['prix_ht_unitaire'], 2, ',', ' '); ?> Ar</span><br>
                                                        <span>TVA unitaire: <?php echo number_format($detail['tva_unitaire'], 2, ',', ' '); ?> %</span>
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                            <p class="details-total-ttc">Total TTC de la commande: <?php echo number_format($commande['total_ttc'], 2, ',', ' '); ?> Ar</p>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
            
            <div class="pagination">
                <?php if ($total_pages > 1): ?>
                    <?php if ($current_page > 1): ?>
                        <a href="?page=<?php echo $current_page - 1; ?>&statut_livraison=<?php echo urlencode($filter_statut_livraison); ?>">Précédent</a>
                    <?php else: ?>
                        <span class="disabled">Précédent</span>
                    <?php endif; ?>

                    <?php
                    $range = 2; // Nombre de pages à afficher autour de la page actuelle
                    $pages_to_show = [];

                    for ($i = 1; $i <= $total_pages; $i++) {
                        if ($i == 1 || $i == $total_pages || ($i >= $current_page - $range && $i <= $current_page + $range)) {
                            $pages_to_show[$i] = true;
                        }
                    }

                    $last_printed_page = 0;
                    foreach (array_keys($pages_to_show) as $page_num) {
                        if ($last_printed_page > 0 && $page_num > $last_printed_page + 1) {
                            echo '<span class="ellipsis">...</span>';
                        }

                        if ($page_num == $current_page) {
                            echo '<span class="current-page">' . $page_num . '</span>';
                        } else {
                            echo '<a href="?page=' . $page_num . '&statut_livraison=' . urlencode($filter_statut_livraison) . '">' . $page_num . '</a>';
                        }
                        $last_printed_page = $page_num;
                    }
                    ?>

                    <?php if ($current_page < $total_pages): ?>
                        <a href="?page=<?php echo $current_page + 1; ?>&statut_livraison=<?php echo urlencode($filter_statut_livraison); ?>">Suivant</a>
                    <?php else: ?>
                        <span class="disabled">Suivant</span>
                    <?php endif; ?>
                <?php endif; ?>
            </div>

        </div>
    </div>
 <?php include('message.php'); ?>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const piecesContainer = document.getElementById('piecesContainer');
            const addNewPieceButton = document.getElementById('addNewPiece');
            const formTitle = document.getElementById('formTitle');
            const submitCommandeButton = document.getElementById('submitCommandeButton');
            const actionTypeCreer = document.getElementById('actionTypeCreer');
            const actionTypeModifier = document.getElementById('actionTypeModifier');
            const modifiedCommandeIdInput = document.getElementById('modifiedCommandeIdInput');
            const cancelEditButton = document.getElementById('cancelEditButton');
            const commentaireInput = document.getElementById('commentaire');
            const listSection = document.querySelector('.list-section');
            let pieceIndex = 0;

            const piecesDisponibles = <?php echo json_encode($pieces_disponibles); ?>;

            function showMessage(message, type) {
                let messageDiv = document.querySelector('.message');
                if (!messageDiv) {
                    messageDiv = document.createElement('div');
                    document.querySelector('.main-content').prepend(messageDiv);
                }
                messageDiv.className = 'message ' + type;
                let iconClass = type === 'success' ? 'fas fa-check-circle' : (type === 'error' ? 'fas fa-times-circle' : 'fas fa-exclamation-triangle');
                messageDiv.innerHTML = `<i class="${iconClass}"></i> ${message}`;
                messageDiv.style.display = 'flex';
            }

            function addNewPieceEntry(pieceData = {}) {
                const newPieceEntry = document.createElement('div');
                newPieceEntry.classList.add('piece-entry');
                newPieceEntry.setAttribute('data-index', pieceIndex);
                
                newPieceEntry.innerHTML = `
                    <div class="form-group flex-1">
                        <label for="filter_piece_${pieceIndex}">Filtrer Pièce:</label>
                        <input type="text" id="filter_piece_${pieceIndex}" class="filter-piece-input" onkeyup="filterPieceOptions(this, ${pieceIndex})" placeholder="Rechercher une pièce...">
                    </div>
                    <div class="form-group flex-1">
                        <label for="id_piece_${pieceIndex}">Pièce:</label>
                        <select name="pieces[${pieceIndex}][id_piece]" id="id_piece_${pieceIndex}" required>
                            <option value="">Sélectionner une pièce</option>
                            ${piecesDisponibles.map(piece => `
                                <option value="${piece.id_piece}"
                                    data-prix="${piece.prix_unitaire}"
                                    data-tva="${piece.tva}"
                                    data-code="${piece.code_piece}"
                                    data-reference="${piece.reference}"
                                    ${pieceData.id_piece == piece.id_piece ? 'selected' : ''}>
                                    ${piece.nom_piece} (Code: ${piece.code_piece}, Réf: ${piece.reference})
                                </option>
                            `).join('')}
                        </select>
                    </div>
                    <div class="form-group flex-1">
                        <label for="quantite_commandee_${pieceIndex}">Quantité:</label>
                        <input type="number" name="pieces[${pieceIndex}][quantite_commandee]" id="quantite_commandee_${pieceIndex}" step="0.5" min="0.5" required value="${pieceData.quantite_commandee || ''}">
                    </div>
                    <div class="form-group flex-1">
                        <label for="prix_ht_unitaire_${pieceIndex}">Prix HT Unitaire:</label>
                        <input type="number" name="pieces[${pieceIndex}][prix_ht_unitaire]" id="prix_ht_unitaire_${pieceIndex}" step="0.01" min="0" required readonly value="${pieceData.prix_ht_unitaire || ''}">
                    </div>
                    <div class="form-group flex-1">
                        <label for="tva_unitaire_${pieceIndex}">TVA Unitaire (%):</label>
                        <input type="number" name="pieces[${pieceIndex}][tva_unitaire]" id="tva_unitaire_${pieceIndex}" step="0.01" min="0" required readonly value="${pieceData.tva_unitaire || ''}">
                    </div>
                    <button type="button" class="remove-piece-btn btn-danger btn-icon">
                        <i class="fas fa-trash"></i>
                    </button>
                `;
                piecesContainer.appendChild(newPieceEntry);

                const selectPiece = newPieceEntry.querySelector(`#id_piece_${pieceIndex}`);
                const prixHtInput = newPieceEntry.querySelector(`#prix_ht_unitaire_${pieceIndex}`);
                const tvaInput = newPieceEntry.querySelector(`#tva_unitaire_${pieceIndex}`);
                
                selectPiece.addEventListener('change', function() {
                    const selectedOption = this.options[this.selectedIndex];
                    prixHtInput.value = selectedOption.value ? selectedOption.dataset.prix : '';
                    tvaInput.value = selectedOption.value ? selectedOption.dataset.tva : '';
                });

                newPieceEntry.querySelector('.remove-piece-btn').addEventListener('click', function() {
                    newPieceEntry.remove();
                    updateRemoveButtonsVisibility();
                });

                if (pieceData.id_piece) {
                    selectPiece.dispatchEvent(new Event('change'));
                }
                pieceIndex++;
                updateRemoveButtonsVisibility();
            }

            window.filterPieceOptions = function(inputElement, index) {
                const filterText = inputElement.value.toLowerCase();
                const selectElement = document.getElementById(`id_piece_${index}`);
                for (let option of selectElement.options) {
                    if (option.value === "") continue;
                    option.style.display = option.textContent.toLowerCase().includes(filterText) ? "" : "none";
                }
                if (selectElement.selectedIndex > 0 && selectElement.options[selectElement.selectedIndex].style.display === "none") {
                    selectElement.value = "";
                }
            };

            function updateRemoveButtonsVisibility() {
                const removeButtons = piecesContainer.querySelectorAll('.remove-piece-btn');
                const display = removeButtons.length <= 1 ? 'none' : 'flex';
                removeButtons.forEach(btn => btn.style.display = display);
            }

            addNewPieceButton.addEventListener('click', () => addNewPieceEntry());

            cancelEditButton.addEventListener('click', function() {
                piecesContainer.innerHTML = '';
                pieceIndex = 0;
                addNewPieceEntry();
                commentaireInput.value = '';
                formTitle.textContent = `Créer une nouvelle commande`;
                submitCommandeButton.textContent = 'Créer la commande';
                actionTypeCreer.value = '1';
                actionTypeModifier.value = '0';
                modifiedCommandeIdInput.value = '';
                cancelEditButton.style.display = 'none';
                showMessage("Formulaire réinitialisé pour la création.", 'info');
            });

            // --- Logique d'initialisation et de gestion de la liste de commandes ---
            function initializeListSectionListeners() {
                // Gérer la modification
                document.querySelectorAll('.edit-order-btn').forEach(button => {
                    button.addEventListener('click', function() {
                        const orderId = this.dataset.id;
                        fetch(`commande.php?action=get_order_details_for_edit&id_commande=${orderId}`)
                            .then(response => response.json())
                            .then(data => {
                                if (data.success) {
                                    piecesContainer.innerHTML = '';
                                    pieceIndex = 0;
                                    data.pieces.forEach(piece => addNewPieceEntry(piece));
                                    commentaireInput.value = data.commentaire;
                                    formTitle.textContent = `Modifier la commande n° ${data.commande_id}`;
                                    submitCommandeButton.innerHTML = '<i class="fas fa-save"></i> Modifier la commande';
                                    actionTypeCreer.value = '0';
                                    actionTypeModifier.value = '1';
                                    modifiedCommandeIdInput.value = data.commande_id;
                                    cancelEditButton.style.display = 'inline-flex';
                                    document.getElementById('commandeForm').scrollIntoView({ behavior: 'smooth' });
                                } else {
                                    showMessage(`Erreur: ${data.message}`, 'error');
                                }
                            }).catch(error => console.error('Erreur fetch (edit):', error));
                    });
                });

                // Gérer la clôture
                document.querySelectorAll('.cloturer-commande-btn').forEach(button => {
                    button.addEventListener('click', function() {
                        const orderId = this.dataset.id;
                        if (confirm(`Êtes-vous sûr de vouloir clôturer la commande n° ${orderId} ?`)) {
                            fetch('commande.php?action=cloturer_commande', {
                                method: 'POST',
                                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                                body: `id_commande=${orderId}`
                            })
                            .then(response => response.json())
                            .then(data => {
                                showMessage(data.message, data.success ? 'success' : 'error');
                                if (data.success) setTimeout(() => window.location.reload(), 1500);
                            }).catch(error => console.error('Erreur fetch (cloture):', error));
                        }
                    });
                });

                // Gérer la génération de bon
                document.querySelectorAll('.generate-bon-btn').forEach(button => {
                    button.addEventListener('click', function() {
                        window.open(`generer_bon_commande.php?id_commande=${this.dataset.id}`, '_blank');
                    });
                });

                // Afficher/cacher les détails
                document.querySelectorAll('.details-toggle-btn').forEach(button => {
                    button.addEventListener('click', function() {
                        const detailsContainer = document.getElementById(`details-${this.dataset.orderId}`);
                        const icon = this.querySelector('i');
                        const isOpening = !detailsContainer.classList.contains('open');

                        document.querySelectorAll('.order-details-container.open').forEach(openContainer => {
                            openContainer.classList.remove('open');
                            openContainer.closest('.order-details-row').style.display = 'none';
                            const otherIcon = openContainer.closest('tr').previousElementSibling.querySelector('.details-toggle-btn i');
                            if(otherIcon) {
                                otherIcon.classList.remove('fa-chevron-up');
                                otherIcon.classList.add('fa-chevron-down');
                            }
                        });

                        if (isOpening) {
                            detailsContainer.classList.add('open');
                            icon.classList.replace('fa-chevron-down', 'fa-chevron-up');
                            this.closest('.order-row').nextElementSibling.style.display = 'table-row';
                        }
                    });
                });
            }
            
            // --- GESTION DE LA PAGINATION AJAX ---
            if (listSection) {
                listSection.addEventListener('click', function(e) {
                    const paginationLink = e.target.closest('.pagination a');
                    if (paginationLink) {
                        e.preventDefault();
                        const url = paginationLink.getAttribute('href');
                        
                        listSection.style.opacity = '0.5'; // Indicateur visuel de chargement

                        fetch(url)
                            .then(response => response.text())
                            .then(html => {
                                const parser = new DOMParser();
                                const doc = parser.parseFromString(html, 'text/html');
                                const newListContent = doc.querySelector('.list-section').innerHTML;
                                
                                listSection.innerHTML = newListContent;
                                history.pushState(null, '', url); // Met à jour l'URL sans recharger
                                initializeListSectionListeners(); // Ré-attache les écouteurs aux nouveaux éléments
                                listSection.scrollIntoView({ behavior: 'smooth' });
                            })
                            .catch(error => {
                                console.error('Erreur lors du chargement de la page:', error);
                                showMessage('Une erreur est survenue lors de la pagination.', 'error');
                            })
                            .finally(() => {
                                listSection.style.opacity = '1';
                            });
                    }
                });
            }

            // Initialisation au chargement de la page
            initializeListSectionListeners();
            if (piecesContainer.children.length === 0) {
                addNewPieceEntry();
            }

            // Burger menu
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Sparkle Effect
            const sparklesContainer = document.querySelector('.sparkles-container');
            for (let i = 0; i < 50; i++) {
                const spark = document.createElement('div');
                spark.classList.add('spark');
                const size = Math.random() * 3 + 1;
                spark.style.width = `${size}px`;
                spark.style.height = `${size}px`;
                spark.style.left = `${Math.random() * 100}%`;
                spark.style.top = `${Math.random() * 100}%`;
                spark.style.animationDelay = `${Math.random() * 2}s`;
                spark.style.setProperty('--random-x', `${(Math.random() - 0.5) * 200}px`);
                sparklesContainer.appendChild(spark);
            }
        });
    </script>
</body>
</html>