<?php
// ASE_Maintenance/clients.php

session_start();
require_once __DIR__ . '/php/config.php'; // Assurez-vous que ce chemin est correct
// Vérification de l'authentification et du rôle Technicien
if (
    ($_SESSION['loggedin'] ?? false) !== true 
    || ($_SESSION['role'] ?? '') !== 'Technicien'
) {
    header('Location: index'); // Rediriger vers la page de connexion (remonte d'un niveau)
    exit();
}


$message = '';
$message_type = '';

// Traitement de la soumission du formulaire d'ajout/modification de client
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_client'])) {
        // Logique d'ajout de client
        $nom_client = filter_input(INPUT_POST, 'nom_client', FILTER_SANITIZE_STRING);
        $contact_client = filter_input(INPUT_POST, 'contact_client', FILTER_SANITIZE_STRING);
        $email_client = filter_input(INPUT_POST, 'email_client', FILTER_SANITIZE_EMAIL);
        $type_client = filter_input(INPUT_POST, 'type_client', FILTER_SANITIZE_STRING);
        
        // Initialiser NIF et STAT à NULL par défaut
        $nif = null; 
        $stat = null; 

        // Si le type de client est "Entreprise", récupérer les valeurs du formulaire
        if ($type_client === 'Entreprise') {
            $nif = filter_input(INPUT_POST, 'nif', FILTER_SANITIZE_STRING);
            $stat = filter_input(INPUT_POST, 'stat', FILTER_SANITIZE_STRING);
            
            // Assurer que les chaînes vides sont converties en NULL avant l'insertion
            // Cela est important si la BDD est configurée pour traiter les chaînes vides différemment des NULL pour les contraintes UNIQUE
            $nif = empty($nif) ? null : $nif;
            $stat = empty($stat) ? null : $stat;
        }

        // Validation basique
        if (empty($nom_client) || empty($contact_client) || empty($type_client)) {
            $message = "Erreur : Le nom, le contact et le type de client sont obligatoires.";
            $message_type = 'error';
        } else {
            try {
                $pdo = getDbConnection();
                $sql = "INSERT INTO clients (nom_client, contact_client, email_client, type_client, nif, stat) VALUES (?, ?, ?, ?, ?, ?)";
                $stmt = $pdo->prepare($sql);
                if ($stmt->execute([$nom_client, $contact_client, $email_client, $type_client, $nif, $stat])) {
                    $message = "Client '" . htmlspecialchars($nom_client) . "' ajouté avec succès !";
                    $message_type = 'success';
                } else {
                    $message = "Erreur lors de l'ajout du client.";
                    $message_type = 'error';
                }
            } catch (PDOException $e) {
                // Vérifier si l'erreur est due à une violation de contrainte unique (code SQLSTATE '23000')
                if ($e->getCode() === '23000') { // SQLSTATE pour violation de contrainte d'intégrité
                    // ATTENTION: La colonne `nom_client` n'a pas de contrainte UNIQUE dans le schéma de table fourni.
                    // Donc, cette condition ne se déclenchera que si vous ajoutez une contrainte UNIQUE sur `nom_client` dans la BDD.
                    // Pour le moment, seuls `email_client`, `nif` et `stat` (s'ils sont uniques dans la BDD) déclencheront cette erreur.
                    if (strpos($e->getMessage(), 'nom_client') !== false) {
                        $message = "Erreur : Un client avec ce nom existe déjà. Veuillez choisir un nom unique.";
                    } elseif (strpos($e->getMessage(), 'email_client') !== false) {
                        $message = "Erreur : Cette adresse e-mail est déjà utilisée par un autre client.";
                    } elseif (strpos($e->getMessage(), 'nif') !== false) {
                        $message = "Erreur : Ce NIF est déjà enregistré pour un autre client.";
                    } elseif (strpos($e->getMessage(), 'stat') !== false) {
                        $message = "Erreur : Ce STAT est déjà enregistré pour un autre client.";
                    } else {
                        $message = "Erreur : Un doublon d'information a été détecté. " . $e->getMessage();
                    }
                } else {
                    $message = "Erreur lors de l'ajout du client : " . $e->getMessage();
                }
                $message_type = 'error';
                error_log("Erreur ajout client: " . $e->getMessage());
            }
        }
    } elseif (isset($_POST['edit_client'])) {
        // Logique de modification de client
        $id_client = filter_input(INPUT_POST, 'id_client', FILTER_SANITIZE_NUMBER_INT);
        $nom_client = filter_input(INPUT_POST, 'nom_client', FILTER_SANITIZE_STRING);
        $contact_client = filter_input(INPUT_POST, 'contact_client', FILTER_SANITIZE_STRING);
        $email_client = filter_input(INPUT_POST, 'email_client', FILTER_SANITIZE_EMAIL);
        
        // On récupère le type existant de la BD pour valider NIF/STAT
        $type_client_existant = '';
        try {
            $pdo = getDbConnection();
            $stmt_type = $pdo->prepare("SELECT type_client FROM clients WHERE id_client = ?");
            $stmt_type->execute([$id_client]);
            $client_data_existing = $stmt_type->fetch(PDO::FETCH_ASSOC);
            if ($client_data_existing) {
                $type_client_existant = $client_data_existing['type_client'];
            }
        } catch (PDOException $e) {
            error_log("Erreur récupération type client existant: " . $e->getMessage());
            // Continuer, mais le type_client_existant sera vide, ce qui fera échouer les validations NIF/STAT si nécessaires
        }

        $nif = null;
        $stat = null;
        if ($type_client_existant === 'Entreprise') {
            $nif = filter_input(INPUT_POST, 'edit_nif', FILTER_SANITIZE_STRING); // Correction: utiliser 'edit_nif' et 'edit_stat'
            $stat = filter_input(INPUT_POST, 'edit_stat', FILTER_SANITIZE_STRING);
            
            // Assurer que les chaînes vides sont converties en NULL avant la mise à jour
            $nif = empty($nif) ? null : $nif;
            $stat = empty($stat) ? null : $stat;
        }

        if (empty($id_client) || empty($nom_client) || empty($contact_client)) {
            $message = "Erreur : Tous les champs obligatoires doivent être remplis pour la modification.";
            $message_type = 'error';
        } else {
            try {
                $pdo = getDbConnection();
                $sql = "UPDATE clients SET nom_client = ?, contact_client = ?, email_client = ?, nif = ?, stat = ? WHERE id_client = ?";
                $stmt = $pdo->prepare($sql);
                // Le type_client n'est plus passé dans le UPDATE directement, car il est considéré fixe
                if ($stmt->execute([$nom_client, $contact_client, $email_client, $nif, $stat, $id_client])) {
                    $message = "Client '" . htmlspecialchars($nom_client) . "' mis à jour avec succès !";
                    $message_type = 'success';
                } else {
                    $message = "Erreur lors de la modification du client.";
                    $message_type = 'error';
                }
            } catch (PDOException $e) {
                 // Vérifier si l'erreur est due à une violation de contrainte unique
                if ($e->getCode() === '23000') { // SQLSTATE pour violation de contrainte d'intégrité
                     // ATTENTION: La colonne `nom_client` n'a pas de contrainte UNIQUE dans le schéma de table fourni.
                    // Donc, cette condition ne se déclenchera que si vous ajoutez une contrainte UNIQUE sur `nom_client` dans la BDD.
                    // Pour le moment, seuls `email_client`, `nif` et `stat` (s'ils sont uniques dans la BDD) déclencheront cette erreur.
                    if (strpos($e->getMessage(), 'nom_client') !== false) {
                        $message = "Erreur : Un autre client avec ce nom existe déjà. Veuillez choisir un nom unique.";
                    } elseif (strpos($e->getMessage(), 'email_client') !== false) {
                        $message = "Erreur : Cette adresse e-mail est déjà utilisée par un autre client.";
                    } elseif (strpos($e->getMessage(), 'nif') !== false) {
                        $message = "Erreur : Ce NIF est déjà enregistré pour un autre client.";
                    } elseif (strpos($e->getMessage(), 'stat') !== false) {
                        $message = "Erreur : Ce STAT est déjà enregistré pour un autre client.";
                    } else {
                        $message = "Erreur : Un doublon d'information a été détecté lors de la modification. " . $e->getMessage();
                    }
                } else {
                    $message = "Erreur lors de la modification du client : " . $e->getMessage();
                }
                $message_type = 'error';
                error_log("Erreur modification client: " . $e->getMessage());
            }
        }
    }
    // Redirection après POST pour éviter la resoumission du formulaire
    $_SESSION['message'] = $message;
    $_SESSION['message_type'] = $message_type;
    header('Location: clients.php');
    exit();
}

// Affichage des messages de session après redirection
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

// --- Récupération des clients pour affichage ---
$clients = [];
try {
    $pdo = getDbConnection();
    // Inclus la colonne date_creation dans la sélection
    $stmt = $pdo->query("SELECT id_client, nom_client, contact_client, email_client, type_client, nif, stat, date_creation FROM clients ORDER BY id_client ASC");
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $message = "Erreur lors du chargement des clients : " . $e->getMessage();
    $message_type = 'error';
    error_log("Erreur chargement clients: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des Clients - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" xintegrity="sha512-SnH5WK+bZxgPHs44uWIX+LLJAJ9/2PkPKZ5QiAj6Ta86w+fsb2TkcmfRyVX3pBnMFcV7oQPJkl9QevSCWr3W6A==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.5/dist/JsBarcode.all.min.js"></script>
    <script src="https://unpkg.com/xlsx/dist/xlsx.full.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.5.25/jspdf.plugin.autotable.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>

    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">


    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18; /* Arrière-plan très sombre */
            --secondary-bg: #1a1a1a; /* Fond du contenu principal (charbon profond) */
            --navbar-bg: rgba(1, 13, 24, 0.95); /* Barre de navigation semi-transparente */
            --text-color: #e0e0e0; /* Texte principal clair */
            --heading-color: #ffda47; /* Jaune doré pour les titres */
            --accent-color: #ffda47; /* Jaune doré pour les accents */
            --border-color: rgba(224, 224, 224, 0.1); /* Bordures très subtiles */
            --form-bg: #2b2b2b; /* Fond des formulaires */
            --input-bg: #3c3c3c; /* Fond des champs de saisie */
            --input-border: #505050; /* Bordure des champs de saisie */
            --table-header-bg: #101010; /* Fond de l'en-tête de tableau */
            --table-row-even-bg: #282828; /* Fond des lignes paires de tableau */

            /* Couleurs des boutons */
            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;

            /* Message colors */
            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            /* Espacements pour les formulaires */
            --form-gap: 15px; /* Espacement entre les éléments de formulaire */
            --form-field-padding: 10px; /* Padding interne des champs */
        }

        /* Styles généraux pour le corps de la page et l'arrière-plan */
        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh; /* Utilise min-height pour permettre au contenu de s'étendre */
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px; /* Base font size agrandie */
        }

        /* Styles pour les étincelles en arrière-plan */
        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg); /* Assurez-vous que cela correspond au body */
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% {
                transform: translateY(-10px) translateX(var(--random-x, 0));
                opacity: 0;
            }
            10% {
                opacity: 1;
            }
            90% {
                opacity: 1;
            }
            100% {
                transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0));
                opacity: 0;
            }
        }

        @keyframes twinkle {
            from {
                opacity: 0.7;
                transform: scale(1);
            }
            to {
                opacity: 1;
                transform: scale(1.2);
            }
        }

        /* Barre de navigation */
        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px; /* Plus de padding */
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5); /* Ombre plus prononcée */
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px; /* Logo légèrement plus grand */
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em; /* Texte de marque plus grand */
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px; /* Espacement plus grand */
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0; /* Plus de padding vertical */
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Taille de police légèrement plus grande */
        }

        .navbar-nav a i {
            margin-right: 10px; /* Espacement entre icône et texte */
            font-size: 1.1em; /* Icônes légèrement plus grandes */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px; /* Soulignement plus épais */
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Menu Burger pour mobile */
        .burger-menu {
            display: none; /* Caché sur desktop */
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0; /* Espacement ajusté */
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) {
            transform: rotate(-45deg) translate(-5px, 6px);
        }

        .burger-menu.open .bar:nth-child(2) {
            opacity: 0;
        }

        .burger-menu.open .bar:nth-child(3) {
            transform: rotate(45deg) translate(-5px, -6px);
        }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0; /* Plus de padding */
            box-shadow: 0 10px 20px rgba(0,0,0,0.3); /* Ombre plus grande */
            z-index: 9;
            transform: translateY(-100%); /* Masqué hors écran */
            transition: transform 0.3s ease-out; /* Animation */
        }

        .navbar-nav.open.slide-in {
            transform: translateY(0);
        }


        .navbar-nav.open li {
            margin: 12px 0; /* Espacement ajusté */
        }

        /* Contenu principal de la page */
        .main-content {
            flex-grow: 1;
            padding: 40px; /* Plus de padding */
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px; /* Contenu agrandi ! */
            width: 95%; /* Prend plus de largeur sur grand écran */
            margin: 30px auto; /* Centre le contenu */
            border-radius: 10px; /* Bordures plus douces */
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6); /* Ombre plus prononcée */
        }

        .container {
            /* Le container interne du code utilisateur est maintenant moins nécessaire avec le main-content élargi,
               mais nous conservons ses styles pour éviter de casser la mise en page existante du code donné par l'utilisateur.
               Ajustons pour qu'il soit transparent et prenne 100% de la largeur du main-content */
            max-width: 100%; /* Prend toute la largeur disponible du main-content */
            margin: 0 auto; /* Centre dans le main-content */
            padding: 0; /* Enlève le padding du container pour laisser le main-content gérer */
            background-color: transparent; /* Rend transparent */
            box-shadow: none; /* Enlève l'ombre du container interne */
        }


        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px; /* Plus d'espace */
            font-size: 3em; /* Titre principal plus grand */
            border-bottom: 4px solid var(--accent-color); /* Bordure plus épaisse */
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title { /* Ajout de .section-title pour cibler les h2 des sections */
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em; /* Titres de section plus grands */
        }

        /* Messages d'alerte/succès/erreur */
        .message {
            padding: 18px; /* Plus de padding */
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em; /* Message plus lisible */
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success {
            background-color: var(--success-bg);
            color: var(--success-text);
            border: 1px solid var(--success-text);
        }

        .message.error {
            background-color: var(--error-bg);
            color: var(--error-text);
            border: 1px solid var(--error-text);
        }
        .message.warning {
            background-color: var(--warning-bg);
            color: var(--warning-text);
            border: 1px solid var(--warning-text);
        }

        /* Styles pour les formulaires et sections */
        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 30px; /* Plus de padding */
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4); /* Ombre plus grande */
            margin-bottom: 40px;
            border: none; /* Retirer la bordure par défaut */
        }

        /* Styles pour les rangées de formulaire (mise en page en grille) */
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr; /* Deux colonnes égales par default */
            gap: var(--form-gap); /* Espacement entre les colonnes et les lignes */
            margin-bottom: var(--form-gap);
        }

        .form-row.full-width {
            grid-template-columns: 1fr; /* Force une seule colonne pour les éléments nécessitant toute la largeur */
        }

        .form-group {
            margin-bottom: 0; /* Réduit la marge par défaut, le gap du form-row gère l'espacement */
        }

        .form-group label {
            display: block;
            margin-bottom: 5px; /* Réduit l'espace sous le label */
            font-weight: bold;
            color: var(--text-color);
            font-size: 1.05em; /* Légèrement plus petit pour compacter */
        }

        .form-group input[type="text"],
        .form-group input[type="email"],
        .form-group input[type="tel"],
        .form-group select {
            width: calc(100% - (var(--form-field-padding) * 2)); /* Ajuste la largeur pour le padding */
            padding: var(--form-field-padding); /* Padding interne réduit */
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 1em; /* Texte des champs légèrement plus petit */
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }

        .form-group input[type="text"]:focus,
        .form-group input[type="email"]:focus,
        .form-group input[type="tel"]:focus,
        .form-group select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 3px rgba(255, 218, 71, 0.3); /* Ombre au focus */
            outline: none;
        }

        .form-group input[type="text"][readonly] {
            background-color: var(--input-bg); /* Garde le thème sombre */
            cursor: not-allowed;
            opacity: 0.7;
        }

        .radio-group {
            display: flex; /* Utilise flexbox pour aligner les radios */
            gap: 20px; /* Espacement entre les options radio */
            align-items: center;
            margin-top: 5px; /* Léger espace au-dessus du groupe */
        }
        .radio-group label {
            margin-bottom: 0; /* Pas de marge sous le label ici */
            display: flex;
            align-items: center;
            font-size: 1em; /* Taille normale */
        }
        .radio-group input[type="radio"] {
            margin-right: 8px; /* Espacement entre radio et texte */
            transform: scale(1); /* Taille normale */
        }

        /* Styles pour le regroupement de champs (NIF/STAT) */
        .form-group-fieldset {
            border: 1px solid var(--input-border);
            padding: 15px;
            border-radius: 8px;
            margin-top: var(--form-gap);
            margin-bottom: var(--form-gap);
            background-color: rgba(0,0,0,0.1); /* Très légère variation de fond */
        }
        .form-group-fieldset legend {
            color: var(--heading-color);
            font-weight: bold;
            padding: 0 10px;
            font-size: 1.1em;
        }
        /* Assure que les champs à l'intérieur du fieldset utilisent la grille */
        .form-group-fieldset .form-row {
            margin-top: 10px;
        }


        /* Styles pour les boutons */
        .btn {
            padding: 12px 25px; /* Plus de padding */
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1.1em; /* Texte des boutons plus grand */
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            text-decoration: none;
            color: white; /* Couleur du texte des boutons */
            margin-top: 15px; /* Espace au-dessus du bouton */
        }

        .btn:hover {
            transform: translateY(-2px); /* Léger effet de survol */
        }

        .btn-primary {
            background-color: var(--btn-primary-bg);
        }
        .btn-primary:hover {
            background-color: var(--btn-primary-hover);
        }

        .btn-success {
            background-color: var(--btn-success-bg);
        }
        .btn-success:hover {
            background-color: var(--btn-success-hover);
        }

        .btn-warning {
            background-color: var(--btn-warning-bg);
            color: #212529; /* Texte foncé pour le jaune */
        }
        .btn-warning:hover {
            background-color: var(--btn-warning-hover);
        }

        .btn-danger {
            background-color: var(--btn-danger-bg);
        }
        .btn-danger:hover {
            background-color: var(--btn-danger-hover);
        }

        .btn-sm {
            padding: 8px 15px; /* Plus de padding pour les petits boutons */
            font-size: 0.9em;
        }

        /* Styles pour les tableaux */
        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg); /* Utilise la couleur du formulaire pour le tableau */
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            min-width: 800px; /* Largeur minimale augmentée pour le tableau */
            border-radius: 8px; /* Coins arrondis pour le tableau */
            overflow: hidden; /* Pour que les coins arrondis fonctionnent avec les bordures */
            margin-top: 25px;
        }

        th, td {
            border: 1px solid var(--input-border); /* Bordures plus foncées */
            padding: 2px; /* Plus de padding */
            text-align: left;
            font-size: 0.7em; /* Texte du tableau plus grand */
            color: var(--text-color);
            vertical-align: middle; /* Centre verticalement le contenu */
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg); /* Lignes paires légèrement différentes */
        }

        tr:hover {
            background-color: #3a3a3a; /* Effet de survol sur les lignes */
            transition: background-color 0.2s ease;
        }

        .table-actions {
            white-space: nowrap;
            gap: 8px; /* Espacement ajusté */
            display: flex;
            justify-content: center; /* Centre les actions dans la cellule */
            align-items: center;
        }

        /* Styles for modals */
        .modal {
            display: none; /* Hidden by default */
            position: fixed; /* Stay in place */
            z-index: 1000; /* Sit on top */
            left: 0;
            top: 0;
            width: 100%; /* Full width */
            height: 100%; /* Full height */
            overflow: auto; /* Enable scroll if needed */
            background-color: rgba(0,0,0,0.6); /* Black w/ more opacity */
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background-color: var(--secondary-bg); /* Fond du modal plus sombre */
            color: var(--text-color);
            /* margin: auto; Ancien style */
            margin: 5% auto; /* Ajuste la marge pour centrer plus efficacement et laisser de l'espace */
            padding: 25px; /* Réduit le padding pour un aspect plus rectangulaire */
            border: 1px solid var(--border-color);
            width: 90%; /* Prend plus de largeur */
            max-width: 800px; /* Augmente la largeur maximale pour un aspect plus rectangulaire */
            border-radius: 10px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.5); /* Ombre plus prononcée */
            position: relative;
            animation: fadeIn 0.3s ease-out;
            box-sizing: border-box; /* Inclut padding et bordure dans la largeur/hauteur */
        }

        .modal-content h2 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 25px;
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 10px;
        }

        @keyframes fadeIn {
            from {opacity: 0; transform: translateY(-20px);}
            to {opacity: 1; transform: translateY(0);}
        }

        .close1-button {
            color: var(--text-color);
            position: absolute;
            right: 20px;
            top: 15px;
            font-size: 32px; /* Plus grande */
            font-weight: bold;
            cursor: pointer;
            transition: color 0.2s ease;
        }

        .close1-button:hover,
        .close1-button:focus {
            color: var(--accent-color);
            text-decoration: none;
        }

        /* Styles de pagination */
        .pagination-controls {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 20px;
            gap: 8px;
        }

        .pagination-button {
            padding: 8px 12px;
            border-radius: 5px;
            background-color: #3f51b5; /* Bleu standard */
            color: white;
            cursor: pointer;
            transition: background-color 0.3s ease;
            font-size: 0.9em;
        }

        .pagination-button:hover:not(:disabled) {
            background-color: #303f9f; /* Bleu plus foncé au survol */
        }

        .pagination-button:disabled {
            background-color: #555;
            cursor: not-allowed;
            opacity: 0.6;
        }

        .pagination-button.active {
            background-color: var(--accent-color); /* Jaune pour la page active */
            color: #1a1a1a;
            font-weight: bold;
        }

        /* Responsive adjustments */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%; /* Reste large */
                padding: 30px;
            }
            h1 {
                font-size: 2.5em;
            }
            h2.section-title {
                font-size: 2em;
            }
            .form-row {
                grid-template-columns: 1fr; /* Passe à une seule colonne sur les écrans plus petits */
            }
            .form-row.full-width {
                grid-template-columns: 1fr; /* S'assure que cela reste 1 colonne */
            }
        }

        @media (max-width: 768px) {
            .navbar-nav {
                display: none; /* Hide by default on small screens */
            }
            .burger-menu {
                display: block; /* Show burger menu on small screens */
            }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto; /* Permet au contenu de prendre toute la largeur disponible */
            }
            h1 {
                font-size: 2.2em;
                margin-bottom: 25px;
            }
            h2.section-title {
                font-size: 1.8em;
                margin-top: 30px;
            }
            th, td {
                padding: 10px;
                font-size: 0.95em;
            }
            table {
                min-width: unset; /* Retire la largeur minimale sur mobile */
            }
            .form-group input, .form-group select {
                width: calc(100% - (var(--form-field-padding) * 2)); /* Maintient le calcul correct */
            }
            .btn {
                width: 100%; /* Boutons pleine largeur */
                justify-content: center; /* Centre le contenu des boutons */
                margin-bottom: 12px;
            }
            .table-actions {
                flex-direction: row; /* Garde les boutons côte à côte si possible */
                flex-wrap: wrap; /* Passe à la ligne si pas assez de place */
                justify-content: center;
                gap: 5px;
            }
            .modal-content {
                width: 95%; /* Prend plus de largeur sur mobile */
                padding: 15px; /* Réduit encore le padding sur mobile */
                margin: 2% auto; /* Ajuste la marge sur mobile */
            }
             .form-group-fieldset {
                padding: 10px;
            }
             .form-group-fieldset legend {
                font-size: 1em;
            }
        }

        @media (max-width: 480px) {
            .main-content {
                padding: 15px;
                margin: 10px auto;
            }
            h1 {
                font-size: 1.8em;
                margin-bottom: 20px;
            }
            h2.section-title {
                font-size: 1.5em;
            }
            .navbar {
                padding: 10px 15px;
            }
            .navbar-brand img {
                height: 35px;
            }
            .navbar-brand span {
                font-size: 1.2em;
            }
            .form-section, .list-section {
                padding: 20px;
            }
            .btn {
                font-size: 1em;
                padding: 10px 20px;
            }
            .table-actions .btn {
                font-size: 0.8em;
                padding: 6px 10px;
            }
             .radio-group {
                flex-direction: column; /* Les radios peuvent passer en colonne sur très petit écran */
                align-items: flex-start;
                gap: 8px;
            }
        }
    </style>
</head>
<body>
    <div class="sparkles-container"></div>

    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="technicien"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="clients" class="active"><i class="fas fa-users"></i> Gestion Clients</a></li>
            <li><a href="vehicules_chauffeurs"><i class="fas fa-car"></i> Véhicules & Chauffeurs</a></li>
            <li><a href="intervention"><i class="fas fa-wrench"></i> Intervention</a></li>
            <li><a href="php/authentification.php?action=logout"> <i style="color: red;" class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="main-content">
        <div class="container">
            <h1>Gestion des Clients</h1>

            <?php if ($message): ?>
                <div class="message <?php echo htmlspecialchars($message_type); ?>">
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>

            <div class="form-section">
                <h2 class="section-title">Ajouter un nouveau client</h2>
                <form action="clients.php" method="POST">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="nom_client">Nom du client :</label>
                            <input type="text" id="nom_client" name="nom_client" required>
                        </div>
                        <div class="form-group">
                            <label for="contact_client">Contact du client :</label>
                            <input type="text" id="contact_client" name="contact_client" required>
                        </div>
                    </div>
                    <div class="form-row full-width">
                        <div class="form-group">
                            <label for="email_client">Email du client (Optionnel) :</label>
                            <input type="email" id="email_client" name="email_client">
                        </div>
                    </div>
                    <div class="form-row full-width">
                        <div class="form-group">
                            <label>Type de client :</label>
                            <div class="radio-group">
                                <label><input type="radio" name="type_client" value="Particulier" checked onclick="toggleClientTypeFields()"> Particulier</label>
                                <label><input type="radio" name="type_client" value="Entreprise" onclick="toggleClientTypeFields()"> Entreprise</label>
                            </div>
                        </div>
                    </div>
                    <fieldset id="entreprise_fields" class="form-group-fieldset" style="display: none;">
                        <legend>Informations Entreprise</legend>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="nif">NIF :</label>
                                <input type="text" id="nif" name="nif">
                            </div>
                            <div class="form-group">
                                <label for="stat">STAT :</label>
                                <input type="text" id="stat" name="stat">
                            </div>
                        </div>
                    </fieldset>
                    <button type="submit" name="add_client" class="btn btn-success">
                        <i class="fas fa-plus-circle"></i> Ajouter Client
                    </button>
                </form>
            </div>

            <hr style="margin: 40px 0; border-top: 1px solid var(--border-color);">

            <div class="list-section">
                <h2 class="section-title">Liste des Clients</h2>
                <div class="overflow-x-auto">
                    <table>
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Nom</th>
                                <th>Contact</th>
                                <th>Email</th>
                                <th>Type</th>
                                <th>NIF</th>
                                <th>STAT</th>
                                <th>Date de Création</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="clientsTableBody">
                            <!-- Les lignes seront chargées par JavaScript -->
                        </tbody>
                    </table>
                </div>
                <!-- Pagination Controls -->
                <div id="clientsPaginationControls" class="pagination-controls"></div>
            </div>
        </div>
    </div>

    <div id="editClientModal" class="modal">
        <div class="modal-content">
            <span class="close1-button" onclick="closeEditModal()">&times;</span>
            <h2>Modifier Client</h2>
            <form action="clients.php" method="POST">
                <input type="hidden" id="edit_id_client" name="id_client">
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_nom_client">Nom du client :</label>
                        <input type="text" id="edit_nom_client" name="nom_client" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_contact_client">Contact du client :</label>
                        <input type="text" id="edit_contact_client" name="contact_client" required>
                    </div>
                </div>
                <div class="form-row full-width">
                    <div class="form-group">
                        <label for="edit_email_client">Email du client (Optionnel) :</label>
                        <input type="email" id="edit_email_client" name="email_client">
                    </div>
                </div>
                <div class="form-row full-width">
                    <div class="form-group">
                        <label for="edit_type_client_display">Type de client :</label>
                        <input type="text" id="edit_type_client_display" name="type_client_display" readonly>
                    </div>
                </div>
                <fieldset id="edit_entreprise_fields" class="form-group-fieldset" style="display: none;">
                    <legend>Informations Entreprise</legend>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_nif">NIF :</label>
                            <input type="text" id="edit_nif" name="edit_nif"> 
                        </div>
                        <div class="form-group">
                            <label for="edit_stat">STAT :</label>
                            <input type="text" id="edit_stat" name="edit_stat"> 
                        </div>
                    </div>
                </fieldset>
                <button type="submit" name="edit_client" class="btn btn-success">
                    <i class="fas fa-save"></i> Enregistrer les modifications
                </button>
            </form>
        </div>
    </div>
<?php include('message.php'); ?>
    <script>
        // Data de tous les clients, chargée une fois depuis PHP
        const allClientsData = <?= json_encode($clients); ?>;
        let currentPageClients = 1;
        const rowsPerPage = 5; // 5 lignes par page, comme demandé

        // --- Helper function to escape HTML characters for display in JS generated HTML ---
        function htmlspecialchars(str) {
            return String(str).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#039;');
        }

        // Script pour gérer l'affichage conditionnel des champs NIF/STAT dans le formulaire d'AJOUT
        function toggleClientTypeFields() {
            const typeClient = document.querySelector('input[name="type_client"]:checked').value;
            const entrepriseFields = document.getElementById('entreprise_fields');
            if (typeClient === 'Entreprise') {
                entrepriseFields.style.display = 'block';
            } else {
                entrepriseFields.style.display = 'none';
                // Optionnel: vider les champs NIF/STAT si on repasse en Particulier
                document.getElementById('nif').value = '';
                document.getElementById('stat').value = '';
            }
        }
        
        // Fonction pour gérer l'affichage des champs NIF/STAT dans le MODAL de MODIFICATION
        function toggleEditClientTypeFields(clientType) {
            const editEntrepriseFields = document.getElementById('edit_entreprise_fields');
            if (clientType === 'Entreprise') {
                editEntrepriseFields.style.display = 'block';
            } else {
                editEntrepriseFields.style.display = 'none';
                // Vider les champs si ce n'est pas une entreprise (au cas où ils étaient pré-remplis)
                document.getElementById('edit_nif').value = '';
                document.getElementById('edit_stat').value = '';
            }
        }

        // Fonction pour ouvrir le modal de modification
        function openEditModal(clientData) {
            const modal = document.getElementById('editClientModal');
            document.getElementById('edit_id_client').value = clientData.id_client;
            document.getElementById('edit_nom_client').value = clientData.nom_client;
            document.getElementById('edit_contact_client').value = clientData.contact_client;
            document.getElementById('edit_email_client').value = clientData.email_client;
            
            // Afficher le type de client en lecture seule
            document.getElementById('edit_type_client_display').value = clientData.type_client;
            
            // Afficher/cacher les champs NIF/STAT selon le type de client
            toggleEditClientTypeFields(clientData.type_client); 
            
            // Pré-remplir NIF et STAT (ils seront vides si ce n'est pas une entreprise)
            document.getElementById('edit_nif').value = clientData.nif || '';
            document.getElementById('edit_stat').value = clientData.stat || '';

            modal.style.display = 'flex'; // Utiliser flex pour centrer
        }

        // Fonction pour fermer le modal de modification
        function closeEditModal() {
            document.getElementById('editClientModal').style.display = 'none';
        }

        // Fermer le modal si l'utilisateur clique en dehors du contenu
        window.onclick = function(event) {
            const modal = document.getElementById('editClientModal');
            if (event.target == modal) {
                modal.style.display = "none";
            }
        }

        // --- Fonctions de pagination et d'affichage du tableau ---

        /**
         * Construit la chaîne HTML pour une seule ligne de client.
         * @param {Object} client L'objet client.
         * @param {number} index L'index de la ligne (pour l'alternance des couleurs).
         * @returns {string} La chaîne HTML de la ligne.
         */
        function buildClientRow(client, index) {
            const rowClass = index % 2 === 0 ? 'bg-gray-800' : 'bg-gray-700';
            const nifDisplay = (client.type_client === 'Particulier' || !client.nif) ? '---------' : htmlspecialchars(client.nif);
            const statDisplay = (client.type_client === 'Particulier' || !client.stat) ? '---------' : htmlspecialchars(client.stat);
            const dateCreationFormatted = new Date(client.date_creation).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit', second: '2-digit' });

            return `
                <tr class="${rowClass}">
                    <td>${htmlspecialchars(client.id_client)}</td>
                    <td>${htmlspecialchars(client.nom_client)}</td>
                    <td>${htmlspecialchars(client.contact_client)}</td>
                    <td>${htmlspecialchars(client.email_client ?? 'N/A')}</td>
                    <td>${htmlspecialchars(client.type_client)}</td>
                    <td>${nifDisplay}</td>
                    <td>${statDisplay}</td>
                    <td>${htmlspecialchars(dateCreationFormatted)}</td>
                    <td class="table-actions">
                        <button class="btn btn-warning btn-sm" onclick="openEditModal(${htmlspecialchars(JSON.stringify(client))})">
                            <i class="fas fa-edit"></i> Modifier
                        </button>
                    </td>
                </tr>
            `;
        }

        /**
         * Affiche les clients pour la page spécifiée.
         * @param {number} page Le numéro de page à afficher.
         */
        function displayClientsPage(page) {
            currentPageClients = page;
            const clientsTableBody = document.getElementById('clientsTableBody');
            clientsTableBody.innerHTML = ''; // Efface le contenu actuel du tableau

            const startIndex = (currentPageClients - 1) * rowsPerPage;
            const endIndex = startIndex + rowsPerPage;
            const clientsToDisplay = allClientsData.slice(startIndex, endIndex);

            if (clientsToDisplay.length === 0) {
                clientsTableBody.innerHTML = '<tr><td colspan="9" class="text-center py-4 text-gray-400">Aucun client enregistré pour le moment.</td></tr>';
            } else {
                clientsToDisplay.forEach((client, index) => {
                    clientsTableBody.innerHTML += buildClientRow(client, index);
                });
            }
            setupClientsPagination(); // Met à jour les contrôles de pagination
        }

        /**
         * Configure les contrôles de pagination pour le tableau des clients.
         */
        function setupClientsPagination() {
            const paginationControls = document.getElementById('clientsPaginationControls');
            paginationControls.innerHTML = ''; // Efface les contrôles existants

            const totalPages = Math.ceil(allClientsData.length / rowsPerPage);

            if (totalPages <= 1) {
                 if (allClientsData.length > 0) {
                    const infoSpan = document.createElement('span');
                    infoSpan.className = 'text-gray-400 text-sm';
                    infoSpan.textContent = `Affichage de ${allClientsData.length} clients.`;
                    paginationControls.appendChild(infoSpan);
                }
                return; // Pas besoin de pagination si 0 ou 1 page
            }

            // Bouton Précédent
            const prevButton = document.createElement('button');
            prevButton.textContent = 'Précédent';
            prevButton.className = 'pagination-button';
            prevButton.disabled = currentPageClients === 1;
            prevButton.onclick = () => displayClientsPage(currentPageClients - 1);
            paginationControls.appendChild(prevButton);

            // Boutons de numéros de page
            const maxPageButtons = 5; // Nombre max de boutons de page à afficher
            let startPage = Math.max(1, currentPageClients - Math.floor(maxPageButtons / 2));
            let endPage = Math.min(totalPages, startPage + maxPageButtons - 1);

            // Ajuster si la fin est trop proche du total pour afficher 5 boutons
            if (endPage - startPage + 1 < maxPageButtons && totalPages > maxPageButtons) {
                startPage = Math.max(1, endPage - maxPageButtons + 1);
            }

            for (let i = startPage; i <= endPage; i++) {
                const pageButton = document.createElement('button');
                pageButton.textContent = i;
                pageButton.className = `pagination-button ${i === currentPageClients ? 'active' : ''}`;
                pageButton.onclick = () => displayClientsPage(i);
                paginationControls.appendChild(pageButton);
            }

            // Bouton Suivant
            const nextButton = document.createElement('button');
            nextButton.textContent = 'Suivant';
            nextButton.className = 'pagination-button';
            nextButton.disabled = currentPageClients === totalPages;
            nextButton.onclick = () => displayClientsPage(currentPageClients + 1);
            paginationControls.appendChild(nextButton);
        }


        // Appeler au chargement de la page pour gérer l'état initial des champs NIF/STAT
        // et pour afficher la première page des clients.
        document.addEventListener('DOMContentLoaded', (event) => {
            toggleClientTypeFields(); // Pour le formulaire d'ajout

            // Script pour le menu burger
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            
            burgerMenu.addEventListener('click', function() {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
                navbarNav.classList.toggle('slide-in'); // Add slide-in class for animation
            });

            // Fermer le menu si un lien est cliqué (pour mobile)
            navbarNav.querySelectorAll('a').forEach(link => {
                link.addEventListener('click', () => {
                    if (navbarNav.classList.contains('open')) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                        navbarNav.classList.remove('slide-in');
                    }
                });
            });

            // Script pour les étincelles
            const sparklesContainer = document.querySelector('.sparkles-container');

            function createSparkle() {
                const spark = document.createElement('div');
                spark.classList.add('spark');
                sparklesContainer.appendChild(spark);

                const size = Math.random() * 3 + 1 + 'px'; // Taille entre 1px et 4px
                spark.style.width = size;
                spark.style.height = size;

                const startX = Math.random() * 100 + 'vw';
                spark.style.left = startX;

                // Délai et durée d'animation aléatoires
                const delay = Math.random() * 2 + 's';
                const duration = Math.random() * 3 + 2 + 's'; // Durée entre 2s et 5s
                spark.style.animationDelay = delay;
                spark.style.animationDuration = duration;

                // Définir une variable CSS pour un petit décalage horizontal aléatoire
                spark.style.setProperty('--random-x', (Math.random() - 0.5) * 200 + 'px'); // Entre -100px et 100px

                // Supprimer l'étincelle après son animation pour éviter l'accumulation
                spark.addEventListener('animationend', () => {
                    spark.remove();
                });
            }

            // Créer de nouvelles étincelles à intervalles réguliers
            setInterval(createSparkle, 100); // Crée une étincelle toutes les 100ms

            // Initialiser l'affichage du tableau des clients avec pagination
            displayClientsPage(currentPageClients);
        });
    </script>
</body>
</html>
