<?php
// ASE_Maintenance/carnet_entretien_vehicule.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Administrateur ou Magasinier
if (
    !isset($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Administrateur'
) {
    header('Location: index');
    exit();
}


$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

$records_per_page = 5; // Nombre d'enregistrements par page (5 lignes par tableau)

// Fonction pour construire la clause WHERE et les paramètres pour les filtres
function buildFilterClauseCarnet($filter_vehicule_sigle, $filter_start_date, $filter_end_date) {
    $sql_where = " WHERE 1=1 ";
    $params = [];

    // Filtrage par sigle du véhicule
    if (!empty($filter_vehicule_sigle) && $filter_vehicule_sigle !== 'all') {
        $sql_where .= " AND v.sigle = ?";
        $params[] = $filter_vehicule_sigle;
    }

    // Filtrage par intervalle de dates si spécifié
    if (!empty($filter_start_date)) {
        $sql_where .= " AND i.date_intervention >= ?";
        $params[] = $filter_start_date;
    }
    if (!empty($filter_end_date)) {
        $sql_where .= " AND i.date_intervention <= ?";
        $params[] = $filter_end_date;
    }

    return ['sql_where' => $sql_where, 'params' => $params];
}

// --- Endpoint AJAX pour récupérer les données filtrées du carnet d'entretien ---
if (isset($_GET['action']) && $_GET['action'] === 'get_filtered_carnet_data') {
    header('Content-Type: application/json');

    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $filter_vehicule_sigle = trim($_GET['filter_vehicule_sigle'] ?? '');
    $filter_start_date = trim($_GET['filter_start_date'] ?? '');
    $filter_end_date = trim($_GET['filter_end_date'] ?? '');

    $offset = ($current_page - 1) * $records_per_page;

    $filter_data = buildFilterClauseCarnet($filter_vehicule_sigle, $filter_start_date, $filter_end_date);
    $sql_where = $filter_data['sql_where'];
    $params = $filter_data['params'];

    try {
        // Pour le comptage total des enregistrements
        $stmt_total = $pdo->prepare("
            SELECT COUNT(DISTINCT i.id_intervention)
            FROM interventions i
            JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            LEFT JOIN clients cli ON i.id_client = cli.id_client
            " . $sql_where
        );
        $stmt_total->execute($params);
        $total_records = $stmt_total->fetchColumn();
        $total_pages = ceil($total_records / $records_per_page);

        // Récupération des données du carnet d'entretien avec pagination
        $stmt_carnet = $pdo->prepare("
            SELECT
                i.id_intervention,
                i.date_intervention,
                i.kilometrage_intervention,
                i.origine_panne,
                i.rapport_intervention,
                v.immatriculation,
                v.sigle,
                v.marque,
                cli.nom_client,
                GROUP_CONCAT(DISTINCT CONCAT(p.nom_piece, ' (', dsp.quantite_sortie, 'x ', dsp.prix_unitaire_ht_au_moment_sortie, ' Ar HT)') SEPARATOR ' ; ') AS details_pieces_resume
            FROM
                interventions i
            JOIN vehicules v ON i.id_vehicule = v.id_vehicule
            LEFT JOIN clients cli ON i.id_client = cli.id_client
            LEFT JOIN sorties_pieces sp ON i.id_intervention = sp.id_intervention
            LEFT JOIN details_sortie_piece dsp ON sp.id_sortie = dsp.id_sortie
            LEFT JOIN piece p ON dsp.id_piece = p.id_piece
            " . $sql_where . "
            GROUP BY i.id_intervention
            ORDER BY i.date_intervention DESC
            LIMIT ? OFFSET ?
        ");
        $params_data = $params;
        $params_data[] = $records_per_page;
        $params_data[] = $offset;
        $stmt_carnet->execute($params_data);
        $carnet_data = $stmt_carnet->fetchAll(PDO::FETCH_ASSOC);

        echo json_encode([
            'success' => true,
            'data' => $carnet_data,
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_records' => $total_records
        ]);

    } catch (PDOException $e) {
        error_log("Erreur PDO lors du chargement du carnet d'entretien: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Erreur de base de données lors du chargement des données : ' . $e->getMessage()]);
    }
    exit();
}

// Récupération de tous les sigles de véhicules pour le filtre (pour l'affichage initial)
try {
    $stmt_vehicule_sigles = $pdo->query("SELECT DISTINCT sigle FROM vehicules WHERE sigle IS NOT NULL ORDER BY sigle ASC");
    $vehicule_sigles = $stmt_vehicule_sigles->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $vehicule_sigles = [];
    $message = "Erreur lors du chargement des sigles de véhicules : " . $e->getMessage();
    $message_type = "error";
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Carnet d'Entretien Véhicule - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">

    <style>
        :root {
            /* Gradient global */
            --gradient-bg: linear-gradient(135deg, #ffffff 65%,  #2ecc71 100%);

            /* Couleurs texte et accents */
            --text-color: #222;
            --heading-color: #002147;
            --accent-color: #2ecc71;   /* vert doux */

            /* Bordures et surfaces */
            --border-color: rgba(0, 0, 0, 0.12);
            --input-bg: #ffffff;
            --input-border: #cfd5db;

            /* Table */
            --table-header-bg: #002147;
            --table-row-even-bg: rgba(0, 0, 0, 0.03);

            /* Boutons */
            --btn-primary-bg: #2563eb;
            --btn-primary-hover: #1d4ed8;
            --btn-success-bg: #22c55e;
            --btn-success-hover: #16a34a;
            --btn-danger-bg: #dc2626;
            --btn-danger-hover: #b91c1c;
            --btn-info-bg: #0891b2;
            --btn-info-hover: #0e7490;

            /* États messages */
            --success-bg: rgba(34, 197, 94, 0.12);
            --success-text: #166534;
            --error-bg: rgba(220, 38, 38, 0.12);
            --error-text: #991b1b;
            --warning-bg: rgba(234, 179, 8, 0.12);
            --warning-text: #92400e;

            --form-gap: 12px;
            --form-field-padding: 10px;
            }

            /* ========== BASE ========== */
            * { box-sizing: border-box; margin: 0; padding: 0; }
            body {
            font-family: 'Inter', sans-serif;
            background: var(--gradient-bg);
            background-attachment: fixed;
            color: var(--text-color);
            font-size: 15px;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            overflow-x: hidden;
            }

            /* Désactive les sparkles pour un rendu pro cohérent */
            .sparkles-container { display: none !important; }

            /* ========== NAVBAR (gradient, comme les autres pages) ========== */
            .navbar {
                display: flex;
                justify-content: space-between;
                align-items: center;
                background: var(--gradient-bg);
                backdrop-filter: blur(8px);
                padding: 15px 30px;
                position: sticky;
                top: 0;
                z-index: 10;
            }

            .navbar img { height: 45px; }
            .navbar-nav { list-style: none; display: flex; gap: 24px; }
            .navbar-nav a {
                color: var(--heading-color);
                text-decoration: none;
                display: flex;
                align-items: center;
                gap: 8px;
                font-weight: 500;
            }
            .navbar-nav a:hover, .navbar-nav a.active { color: var(--accent-color); }

            /* ========== CONTENU PRINCIPAL (carte translucide sur gradient) ========== */
            .main-content {
            flex-grow: 1;
            background: var(--gradient-bg);
            backdrop-filter: blur(6px);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0,0,0,0.06);
            }
        /* Styles pour le menu déroulant (dropdown) */
        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 30px;
            font-size: 2.8em;
            border-bottom: 3px solid var(--accent-color);
            padding-bottom: 12px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 10px;
            margin-top: 40px;
            margin-bottom: 20px;
            font-size: 2em;
        }
        h3.section-title {
            color: var(--heading-color);
            border-bottom: 1px solid var(--accent-color);
            padding-bottom: 6px;
            margin-top: 25px;
            margin-bottom: 15px;
            font-size: 1.6em;
        }

        .message {
            padding: 15px;
            margin-bottom: 25px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.05em;
            display: flex;
            align-items: center;
            gap: 10px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .form-section, .list-section {
            background-color: var(--form-bg);
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 30px;
            border: none;
        }

        .form-group {
            margin-bottom: 12px;
        }

        .form-group label {
            display: block;
            margin-bottom: 4px;
            font-weight: bold;
            color: var(--text-color);
            font-size: 1em;
        }

        .form-group input[type="text"],
        .form-group input[type="email"],
        .form-group input[type="password"],
        .form-group input[type="date"],
        .form-group select {
            width: calc(100% - (var(--form-field-padding) * 2));
            padding: var(--form-field-padding);
            border: 1px solid var(--input-border);
            border-radius: 5px;
            background-color: var(--input-bg);
            color: var(--text-color);
            font-size: 0.95em;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-group input[type="text"]:focus,
        .form-group input[type="email"]:focus,
        .form-group input[type="password"]:focus,
        .form-group input[type="date"]:focus,
        .form-group select:focus {
            border-color: var(--accent-color);
            box-shadow: 0 0 0 2px rgba(255, 218, 71, 0.3);
            outline: none;
        }
        .form-group input[type="text"][readonly] {
            background-color: rgba(var(--input-bg), 0.5);
            cursor: not-allowed;
            opacity: 0.8;
        }

        /* Buttons styles */
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1em;
            font-weight: bold;
            transition: background-color 0.3s ease, transform 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
            color: white;
            margin-top: 10px;
        }
        .btn:hover { transform: translateY(-1px); }
        .btn-primary { background-color: var(--btn-primary-bg); }
        .btn-primary:hover { background-color: var(--btn-primary-hover); }
        .btn-success { background-color: var(--btn-success-bg); }
        .btn-success:hover { background-color: var(--btn-success-hover); }
        .btn-danger { background-color: var(--btn-danger-bg); }
        .btn-danger:hover { background-color: var(--btn-danger-hover); }
        .btn-info { background-color: var(--btn-info-bg); }
        .btn-info:hover { background-color: var(--btn-info-hover); }

        .btn-sm { padding: 6px 12px; font-size: 0.85em; }
        .btn-icon {
            padding: 6px;
            width: 35px;
            height: 35px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        .btn-icon i { margin-right: 0 !important; }

        /* --- Table styles --- */
        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 25px;
            margin-bottom: 25px;
            border-radius: 8px;
            box-shadow: 0 4px 10px rgba(255, 249, 249, 0.4);
 
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 800px; /* Minimum width for the table */
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 8px;
            text-align: left;
            font-size: 0.8em; /* Slightly smaller for this table */
            color: var(--text-color);
            vertical-align: middle;
            white-space: nowrap; /* Prevent wrapping in cells by default */
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 150px; /* Default max-width for table cells */
        }
        td.intervention-report, td.pieces-summary {
            white-space: normal; /* Allow wrapping for long text */
            max-width: 250px; /* Wider for report and pieces */
        }


        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.9em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        tr:hover {
            background-color: #79d158ff;
            transition: background-color 0.2s ease;
        }

        .table-actions {
            display: flex;
            gap: 8px;
            justify-content: center;
            align-items: center;
        }

        /* Pagination styles */
       .pagination {
        display: flex;
        gap: 8px;
        justify-content: center;
        align-items: center;
        margin-top: 14px;
        flex-wrap: wrap;
        }
        .pagination a, .pagination span {
        background: #fff;
        border: 1px solid var(--input-border);
        color: var(--text-color);
        padding: 8px 12px;
        border-radius: 8px;
        text-decoration: none;
        font-weight: 600;
        font-size: .95em;
        }
        .pagination a:hover { background: var(--accent-color); color: #000; border-color: transparent; }
        .pagination .current-page { background: var(--accent-color); color: #000; border-color: transparent; }
        .pagination .disabled { opacity: .5; cursor: not-allowed; }
        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 25px;
            }
            h1 { font-size: 2.3em; }
            h2.section-title, h3.section-title { font-size: 1.8em; }
            .form-group input[type="text"],
            .form-group input[type="email"],
            .form-group input[type="password"],
            .form-group input[type="date"],
            .form-group select {
                width: 100%;
            }
            .dropdown-menu {
                position: static;
                box-shadow: none;
                background-color: transparent;
                width: 100%;
                padding-left: 20px;
                border-radius: 0;
            }
            th, td {
                font-size: 0.7em;
                padding: 6px;
                max-width: unset;
                white-space: normal;
            }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title, h3.section-title { font-size: 1.6em; margin-top: 25px; }
            table { min-width: unset; }
            .btn { width: 100%; justify-content: center; margin-bottom: 10px; padding: 8px 15px; font-size: 0.95em; }
            .pagination {
                flex-wrap: wrap;
                gap: 5px;
            }
            .pagination a, .pagination span {
                padding: 6px 10px;
                font-size: 0.9em;
            }
            .dropdown > a {
                cursor: pointer;
            }
            .button-group-modal {
                flex-direction: column;
                gap: 10px;
            }
            /* Formulaire 2 champs par ligne sur mobile -> 1 champ par ligne */
            .flex.flex-wrap.items-end.justify-between.mb-4.gap-4 {
                display: block; /* Override flex */
            }
            .flex.flex-wrap.items-end.justify-between.mb-4.gap-4 > div { /* Correction ici */
                width: 100% !important;
                padding-left: 0 !important;
                padding-right: 0 !important;
            }
            .form-section, .list-section { padding: 15px; margin-bottom: 20px;}
        }

        @media (max-width: 480px) {
            .main-content { padding: 10px; margin: 10px auto; }
            h1 { font-size: 1.6em; margin-bottom: 15px; }
            h2.section-title, h3.section-title { font-size: 1.4em; }
            .navbar { padding: 8px 10px; }
            .navbar-brand img { height: 30px; }
            .navbar-brand span { font-size: 1.1em; }
            .btn { font-size: 0.9em; padding: 8px 15px; }
        }

        /* --- MODAL STYLES --- */
        .modal1 {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.7);
            justify-content: center;
            align-items: center;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background-color: var(--form-bg);
            margin: auto;
            padding: 20px 25px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 95%;
            max-width: 700px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            animation: slideIn 0.3s forwards;
        }

        .modal1.open {
            display: flex;
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        .close1-button {
            color: var(--text-color);
            position: absolute;
            top: 12px;
            right: 20px;
            font-size: 26px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close1-button:hover,
        .close1-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        /* End MODAL STYLES */
        .modal1-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(164, 164, 164, 0.7);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 1000;
        }
        .modal1-content {
            background-color: white;
            padding: 30px;
            border-radius: 10px;
            max-width: 900px;
            width: 90%;
            border: 1px solid var(--input-border);
            box-shadow: white;
            position: relative;
        }
        .close1-btn {
            position: absolute;
            top: 15px;
            right: 25px;
            color: var(--text-color);
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }
        .close1-btn:hover {
            color: var(--accent-color);
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div><img src="img/logo_ase.png" alt="Logo"></div>
        <ul class="navbar-nav">
            <li><a href="tableau_bord"><i class="fas fa-tachometer-alt"></i> Tableau de bord</a></li>
            <li><a href="details_sorties_globales"><i class="fas fa-gauge-high"></i> Admin legacy</a></li>
            <li><a href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports</a></li>
            <li><a class="active" href="carnet_entretien_vehicule"><i class="fas fa-truck"></i> Flotte</a></li>
            <li><a href="rapport_pneumatique"><i class="fas fa-history"></i> Détails pneumatique</a></li>
            <li><a href="achats_effectues"><i class="fas fa-boxes-packing"></i> Achats effectués</a></li>
            <li><a href="sorties_effectuees"><i class="fas fa-truck-loading"></i> Sorties effectuées</a></li>
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="sparkles-container"></div>
    <div class="main-content">
        <h1 id="page-title">Carnet d'Entretien par Véhicule</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <!-- Section de filtrage -->
        <div class="form-section">
            <h2 class="section-title">Filtres du Carnet d'Entretien</h2>
            <div class="flex flex-wrap items-end justify-between mb-4 gap-4">
                <div class="form-group flex-grow mb-0">
                    <label for="filter_vehicule_sigle" style="background-color: white; color: black;">Sélectionner un Véhicule (Sigle):</label>
                    <select id="filter_vehicule_sigle" class="w-full" style="background-color: white; color: black;">
                        <option value="all">Tous les véhicules</option>
                        <?php foreach ($vehicule_sigles as $sigle): ?>
                            <option value="<?php echo htmlspecialchars($sigle['sigle']); ?>"><?php echo htmlspecialchars($sigle['sigle']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group flex-grow mb-0">
                    <label for="filter_start_date" style="background-color: white; color: black;">Date de Début:</label>
                    <input type="date" id="filter_start_date" class="w-full" style="background-color: white; color: black;">
                </div>
                <div class="form-group flex-grow mb-0">
                    <label for="filter_end_date" style="background-color: white; color: black;">Date de Fin:</label>
                    <input type="date" id="filter_end_date" class="w-full" style="background-color: white; color: black;">
                </div>
                <div class="mb-0" >
                    <button type="button" class="btn btn-primary btn-sm" id="apply_filters_carnet"><i class="fas fa-filter"></i> Appliquer les filtres</button>
                </div>
                 <div class="mb-0">
                    <button type="button" class="btn btn-success btn-sm" id="generate_carnet_pdf_btn"><i class="fas fa-book-medical"></i> Générer Carnet PDF</button>
                </div>
            </div>
        </div>

        <!-- Section pour la liste des entretiens -->
        <div class="list-section">
            <h2 class="section-title">Historique des Interventions</h2>
            <div class="table-responsive">
                <table id="carnetEntretienTable">
                    <thead>
                        <tr>
                            <th>Date Interv.</th>
                            <th>Véhicule (Immatriculation / Sigle)</th>
                            <th>Client</th>
                            <th>Kilométrage</th>
                            <th>Origine Panne</th>
                            <th>Rapport Interv.</th>
                            <th>Pièces Utilisées</th>
                        </tr>
                    </thead>
                    <tbody id="carnetEntretienTableBody">
                        <!-- Les données seront chargées ici via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="carnetEntretienPagination">
                <!-- La pagination sera générée ici -->
            </div>
        </div>
        <h1 id="page-title">Pointage Kilométrage et Voyages & facturation kilometrage</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <div class="dashboard-section">
            <div class="flex justify-between items-center mb-6">
                <h2 class="section-title !mb-0 !mt-0 !pb-0 !border-0">Liste des pointages</h2>
                <button id="openModalBtn" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded transition duration-200">
                    <i class="fas fa-plus mr-2"></i>Nouveau Pointage
                </button>
            </div>

            <div class="flex items-center mb-6 space-x-4 mt-6">
                <input type="month" id="filterMonth" class="bg-white-800 text-black p-2 rounded-lg border border-gray-700 focus:ring-blue-500 focus:border-blue-500">
                <input type="text" id="filterSearch" placeholder="Rechercher par immatriculation..." class="bg-white-800 text-black p-2 rounded-lg border border-gray-700 focus:ring-blue-500 focus:border-blue-500 w-64">
            </div>

            <div class="table-responsive">
                <table id="pointageTable" class="min-w-full">
                    <thead>
                        <tr>
                            <th scope="col">Véhicule</th>
                            <th scope="col">Chauffeur</th>
                            <th scope="col">Date Pointage</th>
                            <th scope="col">Début Mois (Km)</th>
                            <th scope="col">Fin Mois (Km)</th>
                            <th scope="col">Kilométrage Total</th>
                            <th scope="col">Voyages (Estimé)</th>
                            <th scope="col">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="tableBody">
                        </tbody>
                </table>
            </div>

            <div class="pagination" id="pagination-container">
                <button id="prevPageBtn" class="py-1 px-3 rounded-lg text-sm bg-gray-700 hover:bg-gray-600 text-gray-300 disabled:opacity-50 disabled:cursor-not-allowed">Précédent</button>
                <div id="paginationButtons" class="flex space-x-1"></div>
                <button id="nextPageBtn" class="py-1 px-3 rounded-lg text-sm bg-gray-700 hover:bg-gray-600 text-gray-300 disabled:opacity-50 disabled:cursor-not-allowed">Suivant</button>
            </div>
            <br>
                    <!-- ===================== FACTURATION ===================== -->
        <div class="list-section">
                        <!-- Section Facturation -->
                <div class="p-6 bg-gradient-to-br from-white to-gray-50 rounded-xl shadow-lg border border-gray-200">
                    <h2 class="text-2xl font-bold mb-6 text-gray-800 flex items-center gap-2">
                        <i class="fas fa-file-invoice text-blue-500"></i> Générer une Facture
                    </h2>
                    <form id="facturationForm" class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        
                        <!-- Sélection Client -->
                        <div>
                            <label for="factureClient" class="block text-sm font-semibold text-gray-700 mb-1">Client</label>
                            <select id="factureClient" name="id_client" 
                                class="w-full p-3 border border-gray-300 rounded-lg shadow-sm focus:ring-2 focus:ring-blue-400 focus:border-blue-400 transition">
                                <!-- Options clients chargées dynamiquement -->
                            </select>
                        </div>

                        <!-- Mois de facturation -->
                        <div>
                            <label for="moisFacturation" class="block text-sm font-semibold text-gray-700 mb-1">Mois de facturation</label>
                            <input type="month" id="moisFacturation" name="mois_facturation" required
                                class="w-full p-3 border border-gray-300 rounded-lg shadow-sm focus:ring-2 focus:ring-blue-400 focus:border-blue-400 transition">
                        </div>

                        <!-- Tarif au km -->
                        <div>
                            <label for="tarifKm" class="block text-sm font-semibold text-gray-700 mb-1">Tarif au km (MGA)</label>
                            <input type="number" id="tarifKm" name="tarif_km" step="0.01" required
                                class="w-full p-3 border border-gray-300 rounded-lg shadow-sm focus:ring-2 focus:ring-blue-400 focus:border-blue-400 transition">
                        </div>

                        <!-- TVA -->
                        <div>
                            <label for="tva" class="block text-sm font-semibold text-gray-700 mb-1">TVA (%)</label>
                            <input type="number" id="tva" name="tva" step="0.01" required
                                class="w-full p-3 border border-gray-300 rounded-lg shadow-sm focus:ring-2 focus:ring-blue-400 focus:border-blue-400 transition">
                        </div>

                        <!-- Bouton -->
                        <div class="md:col-span-2 flex justify-end">
                            <button type="submit" 
                                class="px-6 py-3 bg-blue-600 text-white font-semibold rounded-lg shadow-md hover:bg-blue-700 hover:shadow-lg transform hover:-translate-y-0.5 transition">
                                <i class="fas fa-check-circle mr-2"></i> Sauvegarder
                            </button>
                        </div>
                    </form>
                </div>



            <!-- Tableau des véhicules du client -->
            <div class="table-responsive">
                <table id="factureVehiculesTable">
                    <thead>
                        <tr>
                            <th>Véhicule</th>
                            <th>Kilométrage</th>
                            <th>Montant HT</th>
                            <th>TVA</th>
                            <th>Montant TTC</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr><td colspan="7" class="text-center">Sélectionnez un client pour voir ses véhicules</td></tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- ===================== HISTORIQUE FACTURATION ===================== -->
       <div class="list-section">
            <h2 class="section-title">Historique des Facturations</h2>
            <div class="table-responsive">
                <table id="historiqueFacturesTable">
                    <thead>
                        <tr>
                            <th>Numéro de facture</th>
                            <th>Client</th>
                            <th>Mois facturé</th>          <!-- 🆕 -->
                            <th>Date de facturation</th>   <!-- 🆕 -->
                            <th>Kilométrage total</th>
                            <th>Montant TT HT</th>
                            <th>Montant TT TVA</th>
                            <th>Montant TT TTC</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td colspan="9" class="text-center">Aucune facture enregistrée</td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        </div>
    </div>
        <div id="pointageModal" class="modal1-overlay hidden">
        <div class="modal1-content">
            <span class="close1-btn" id="closeModalBtn">&times;</span>
            <h3 class="section-title" id="modalTitle">Ajouter un Pointage</h3>
            <form id="pointageForm">
                <input type="hidden" id="pointageId" name="id_pointage">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label for="vehicule" class="block text-sm font-medium text-black-400">Véhicule</label>
                        <select id="vehicule" name="id_vehicule" class="mt-1 block w-full bg-white-700 border border-gray-600 rounded-md shadow-sm p-2 text-black focus:ring-blue-500 focus:border-blue-500" required>
                            </select>
                    </div>
                    <div>
                        <label for="chauffeur" class="block text-sm font-medium text-black-400">Chauffeur</label>
                        <select id="chauffeur" name="id_chauffeur" class="mt-1 block w-full bg-white-700 border border-gray-600 rounded-md shadow-sm p-2 text-black focus:ring-blue-500 focus:border-blue-500">
                            </select>
                    </div>
                    <div class="col-span-2">
                        <label for="datePointage" class="block text-sm font-medium text-black-400">Date Pointage</label>
                        <input type="date" id="datePointage" name="date_pointage" class="mt-1 block w-full bg-white-700 border border-gray-600 rounded-md shadow-sm p-2 text-black focus:ring-blue-500 focus:border-blue-500" required>
                    </div>
                    <div class="col-span-2">
                        <p class="text-sm font-bold text-black-400 mb-2">Saisie du kilométrage :</p>
                        <div class="flex space-x-4">
                            <div class="flex-1">
                                <label for="debutMois" class="block text-sm text-black-500">Début Mois (Km)</label>
                                <input type="number" id="debutMois" name="kilometrage_debut_mois" class="mt-1 block w-full bg-white-700 border border-gray-600 rounded-md shadow-sm p-2 text-black focus:ring-blue-500 focus:border-blue-500">
                            </div>
                            <div class="flex-1">
                                <label for="finMois" class="block text-sm text-black-500">Fin Mois (Km)</label>
                                <input type="number" id="finMois" name="kilometrage_fin_mois" class="mt-1 block w-full bg-white-700 border border-gray-600 rounded-md shadow-sm p-2 text-black focus:ring-blue-500 focus:border-blue-500">
                            </div>
                        </div>
                    </div>
                    <div class="col-span-2 text-center text-black-400 font-bold">OU</div>
                    <div class="col-span-2">
                        <p class="text-sm font-bold text-black-400 mb-2">Saisie par voyages :</p>
                        <div class="flex space-x-4">
                            <div class="flex-1">
                                <label for="nombreVoyages" class="block text-sm text-black-500">Nombre de Voyages</label>
                                <input type="number" id="nombreVoyages" name="nombre_voyages" class="mt-1 block w-full bg-white-700 border border-gray-600 rounded-md shadow-sm p-2 text-black focus:ring-blue-500 focus:border-blue-500">
                            </div>
                            <div class="flex-1">
                                <label for="kilometrageParVoyage" class="block text-sm text-black-500">Km par Voyage (estimé)</label>
                                <input type="number" id="kilometrageParVoyage" name="kilometrage_par_voyage_estime" class="mt-1 block w-full bg-white-700 border border-gray-600 rounded-md shadow-sm p-2 text-black focus:ring-blue-500 focus:border-blue-500">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="mt-6 flex justify-end">
                    <button type="button" id="cancelModalBtn" class="bg-gray-600 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded mr-2 transition duration-200">Annuler</button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded transition duration-200">Enregistrer</button>
                </div>
            </form>
        </div>
    </div>

<?php include('message.php'); ?>
    <script>
        // Fonction globale pour afficher les messages (réutilisée)
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        // Fonction pour générer les liens de pagination (réutilisée)
        function generatePagination(container, currentPage, totalPages, loadFunction) {
            container.innerHTML = '';
            if (totalPages <= 1) return;

            const maxPagesToShow = 5;
            let startPage = Math.max(1, currentPage - Math.floor(maxPagesToShow / 2));
            let endPage = Math.min(totalPages, startPage + maxPagesToShow - 1);

            if (endPage - startPage + 1 < maxPagesToShow) {
                startPage = Math.max(1, endPage - maxPagesToShow + 1);
            }

            const appendLink = (page, text, className = '') => {
                const link = document.createElement('a');
                link.href = '#'; // Prevent actual navigation
                link.dataset.page = page;
                link.textContent = text;
                link.classList.add('px-3', 'py-1', 'rounded-md');
                if (className) link.classList.add(className);
                if (page === currentPage) link.classList.add('bg-blue-600', 'text-white'); // Active page style
                // Disable links for pages out of bounds or current page
                if (page < 1 || page > totalPages || page === currentPage) link.classList.add('pointer-events-none', 'opacity-50');

                link.addEventListener('click', (e) => {
                    e.preventDefault();
                    if (!link.classList.contains('disabled') && !link.classList.contains('pointer-events-none')) {
                        loadFunction(parseInt(link.dataset.page));
                    }
                });
                container.appendChild(link);
            };

            appendLink(currentPage - 1, 'Précédent', 'prev-page-btn');

            if (startPage > 1) {
                appendLink(1, '1');
                if (startPage > 2) {
                    const ellipsis = document.createElement('span');
                    ellipsis.textContent = '...';
                    ellipsis.classList.add('px-3', 'py-1');
                    container.appendChild(ellipsis);
                }
            }

            for (let i = startPage; i <= endPage; i++) {
                appendLink(i, i.toString(), i === currentPage ? 'current-page' : '');
            }

            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    const ellipsis = document.createElement('span');
                    ellipsis.textContent = '...';
                    ellipsis.classList.add('px-3', 'py-1');
                    container.appendChild(ellipsis);
                }
                appendLink(totalPages, totalPages.toString());
            }

            appendLink(currentPage + 1, 'Suivant', 'next-page-btn');
        }

        // Fonction pour formater les nombres en monnaie (si besoin)
        function formatCurrency(amount) {
            return parseFloat(amount).toLocaleString('fr-FR', { style: 'currency', currency: 'MGA' });
        }

        // --- Fonctions de chargement des données pour le carnet d'entretien ---
        let currentPageCarnet = 1; // Variable pour la page courante
        async function loadCarnetEntretienData(page = 1) {
            const filterVehiculeSigle = document.getElementById('filter_vehicule_sigle').value;
            const filterStartDate = document.getElementById('filter_start_date').value;
            const filterEndDate = document.getElementById('filter_end_date').value;

            try {
                let queryString = `action=get_filtered_carnet_data&page=${page}`;
                if (filterVehiculeSigle) queryString += `&filter_vehicule_sigle=${encodeURIComponent(filterVehiculeSigle)}`;
                if (filterStartDate) queryString += `&filter_start_date=${encodeURIComponent(filterStartDate)}`;
                if (filterEndDate) queryString += `&filter_end_date=${encodeURIComponent(filterEndDate)}`;

                const response = await fetch(`carnet_entretien_vehicule.php?${queryString}`);
                const data = await response.json();

                const carnetEntretienTableBody = document.getElementById('carnetEntretienTableBody');
                const carnetEntretienPagination = document.getElementById('carnetEntretienPagination');
                carnetEntretienTableBody.innerHTML = ''; // Clear existing data

                if (data.success && data.data.length > 0) {
                    data.data.forEach(intervention => {
                        const row = `
                            <tr>
                                <td>${new Date(intervention.date_intervention).toLocaleDateString('fr-FR')}</td>
                                <td>${intervention.immatriculation} (${intervention.sigle || 'N/A'})</td>
                                <td>${intervention.nom_client || 'N/A'}</td>
                                <td>${intervention.kilometrage_intervention || 'N/A'} km</td>
                                <td class="intervention-report" title="${intervention.origine_panne}">${intervention.origine_panne ? intervention.origine_panne.substring(0, 100) + (intervention.origine_panne.length > 100 ? '...' : '') : 'N/A'}</td>
                                <td class="intervention-report" title="${intervention.rapport_intervention}">${intervention.rapport_intervention ? intervention.rapport_intervention.substring(0, 100) + (intervention.rapport_intervention.length > 100 ? '...' : '') : 'N/A'}</td>
                                <td class="pieces-summary" title="${intervention.details_pieces_resume}">${intervention.details_pieces_resume ? intervention.details_pieces_resume.substring(0, 100) + (intervention.details_pieces_resume.length > 100 ? '...' : '') : 'Aucune'}</td>
                            </tr>
                        `;
                        carnetEntretienTableBody.insertAdjacentHTML('beforeend', row);
                    });
                    generatePagination(carnetEntretienPagination, data.current_page, data.total_pages, loadCarnetEntretienData);
                } else {
                    carnetEntretienTableBody.innerHTML = `<tr><td colspan="7" class="text-center">${data.message || 'Aucun entretien trouvé avec les filtres appliqués.'}</td></tr>`;
                    carnetEntretienPagination.innerHTML = ''; // Clear pagination if no data
                }
                currentPageCarnet = page; // Mettre à jour la page courante après le chargement
            } catch (error) {
                console.error('Erreur AJAX lors du chargement du carnet d\'entretien:', error);
                showMessage('Erreur technique lors du chargement des données du carnet d\'entretien.', 'error');
            }
        }

        document.addEventListener('DOMContentLoaded', function() {

            // Filtres et boutons
            const filterVehiculeSigleSelect = document.getElementById('filter_vehicule_sigle');
            const filterStartDateInput = document.getElementById('filter_start_date');
            const filterEndDateInput = document.getElementById('filter_end_date');
            const applyFiltersBtn = document.getElementById('apply_filters_carnet');
            const generateCarnetPdfBtn = document.getElementById('generate_carnet_pdf_btn');



            // Événements pour les filtres et le bouton d'application
            applyFiltersBtn.addEventListener('click', () => loadCarnetEntretienData(1)); // Toujours revenir à la première page
            filterVehiculeSigleSelect.addEventListener('change', () => loadCarnetEntretienData(1));
            filterStartDateInput.addEventListener('change', () => loadCarnetEntretienData(1));
            filterEndDateInput.addEventListener('change', () => loadCarnetEntretienData(1));

            // Événement pour le bouton "Générer Carnet PDF"
            generateCarnetPdfBtn.addEventListener('click', () => {
                const filterVehiculeSigle = document.getElementById('filter_vehicule_sigle').value;
                const filterStartDate = document.getElementById('filter_start_date').value;
                const filterEndDate = document.getElementById('filter_end_date').value;

                let queryString = ``;
                if (filterVehiculeSigle !== 'all') queryString += `filter_vehicule_sigle=${encodeURIComponent(filterVehiculeSigle)}&`;
                if (filterStartDate) queryString += `filter_start_date=${encodeURIComponent(filterStartDate)}&`;
                if (filterEndDate) queryString += `filter_end_date=${encodeURIComponent(filterEndDate)}&`;

                // Supprimer le dernier '&' s'il existe
                if (queryString.endsWith('&')) {
                    queryString = queryString.slice(0, -1);
                }

                // Ouvrir la page de génération de PDF dans un nouvel onglet/fenêtre
                window.open(`generate_carnet_pdf.php?${queryString}`, '_blank');
            });
            
            // Chargement initial des données
            loadCarnetEntretienData(1); // Charger la première page au démarrage
            loadHistoriqueFactures();
            // Script pour la page de pointage des véhicules
        const pointageTableBody = document.getElementById('tableBody');
        const pointageForm = document.getElementById('pointageForm');
        const pointageModal = document.getElementById('pointageModal');
        const openModalBtn = document.getElementById('openModalBtn');
        const closeModalBtn = document.getElementById('closeModalBtn');
        const cancelModalBtn = document.getElementById('cancelModalBtn');
        const modalTitle = document.getElementById('modalTitle');
        const vehiculeSelect = document.getElementById('vehicule');
        const chauffeurSelect = document.getElementById('chauffeur');
        const filterMonth = document.getElementById('filterMonth');
        const filterSearch = document.getElementById('filterSearch');

        let currentPage = 1;
        const pageSize = 10;
        let totalEntries = 0;
        let pointageData = [];

        // Récupérer les données initiales
        fetchData();

        openModalBtn.addEventListener('click', () => {
            modalTitle.textContent = 'Ajouter un Pointage';
            pointageForm.reset();
            document.getElementById('pointageId').value = '';
            pointageModal.classList.remove('hidden');
            loadVehiculesAndChauffeurs();
        });

        closeModalBtn.addEventListener('click', () => pointageModal.classList.add('hidden'));
        cancelModalBtn.addEventListener('click', () => pointageModal.classList.add('hidden'));

        pointageForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const data = Object.fromEntries(formData.entries());

            // Validation de la saisie
            const hasKm = (data.kilometrage_debut_mois !== '' && data.kilometrage_fin_mois !== '');
            const hasVoyages = (data.nombre_voyages !== '' && data.kilometrage_par_voyage_estime !== '');
            
            if (hasKm && hasVoyages) {
                alert("Vous ne pouvez pas saisir à la fois le kilométrage et le nombre de voyages.");
                return;
            }
            if (!hasKm && !hasVoyages) {
                alert("Veuillez saisir soit le kilométrage, soit le nombre de voyages.");
                return;
            }

            const action = data.id_pointage ? 'update_pointage' : 'add_pointage';
            
            fetch('php/api_pointage_vehicule.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ action: action, ...data })
            })
            .then(response => response.json())
            .then(result => {
                if (result.success) {
                    alert('Pointage enregistré avec succès.');
                    pointageModal.classList.add('hidden');
                    fetchData();
                } else {
                    alert('Erreur: ' + result.error);
                }
            })
            .catch(error => {
                console.error('Erreur:', error);
                alert('Une erreur est survenue lors de l\'enregistrement.');
            });
        });

        function fetchData() {
            const month = filterMonth.value;
            const search = filterSearch.value;

            fetch(`php/api_pointage_vehicule.php?action=fetch_data&month=${month}&search=${search}`)
                .then(response => response.json())
                .then(data => {
                    pointageData = data.data;
                    totalEntries = pointageData.length;
                    renderTable();
                });
        }

        function renderTable() {
            pointageTableBody.innerHTML = '';
            const start = (currentPage - 1) * pageSize;
            const end = Math.min(start + pageSize, totalEntries);
            const paginatedData = pointageData.slice(start, end);

            paginatedData.forEach(item => {
                let kilometrageTotal = '';
                let voyagesInfo = '';

                if (item.kilometrage_debut_mois !== null && item.kilometrage_fin_mois !== null) {
                    kilometrageTotal = (item.kilometrage_fin_mois - item.kilometrage_debut_mois).toLocaleString();
                } else if (item.nombre_voyages !== null && item.kilometrage_par_voyage_estime !== null) {
                    kilometrageTotal = (item.nombre_voyages * item.kilometrage_par_voyage_estime).toLocaleString();
                    voyagesInfo = `${item.nombre_voyages} voyages x ${item.kilometrage_par_voyage_estime} Km`;
                }

                const row = `
                    <tr class="hover:bg-gray-700 transition duration-200">
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">${item.vehicule_immatriculation} (${item.vehicule_marque})</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${item.nom_chauffeur || 'N/A'}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${item.date_pointage}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${item.kilometrage_debut_mois !== null ? item.kilometrage_debut_mois.toLocaleString() : 'N/A'}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${item.kilometrage_fin_mois !== null ? item.kilometrage_fin_mois.toLocaleString() : 'N/A'}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-bold text-blue-400">${kilometrageTotal} Km</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">${voyagesInfo || 'N/A'}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm">
                            <button onclick="editPointage(${item.id_pointage})" class="text-blue-500 hover:text-blue-700 mr-3"><i class="fas fa-edit"></i></button>
                        </td>
                    </tr>
                `;
                pointageTableBody.innerHTML += row;
            });

            updatePaginationInfo(start, end);
        }

        function updatePaginationInfo(start, end) {
            const totalPages = Math.ceil(totalEntries / pageSize);
            
            document.getElementById('prevPageBtn').disabled = currentPage === 1;
            document.getElementById('nextPageBtn').disabled = currentPage === totalPages || totalEntries === 0;

            const paginationButtonsContainer = document.getElementById('paginationButtons');
            paginationButtonsContainer.innerHTML = '';

            for (let i = 1; i <= totalPages; i++) {
                const button = document.createElement('button');
                button.textContent = i;
                button.className = `py-1 px-3 rounded-lg text-sm ${i === currentPage ? 'bg-blue-600 text-white' : 'bg-gray-700 hover:bg-gray-600 text-gray-300'}`;
                button.onclick = () => {
                    currentPage = i;
                    renderTable();
                };
                paginationButtonsContainer.appendChild(button);
            }
        }
        
        document.getElementById('prevPageBtn').addEventListener('click', () => {
            if (currentPage > 1) {
                currentPage--;
                renderTable();
            }
        });

        document.getElementById('nextPageBtn').addEventListener('click', () => {
            if (currentPage < Math.ceil(totalEntries / pageSize)) {
                currentPage++;
                renderTable();
            }
        });

        window.editPointage = function(id) {
            fetch(`php/api_pointage_vehicule.php?action=get_pointage_by_id&id=${id}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const item = data.data;
                        document.getElementById('modalTitle').textContent = 'Modifier un Pointage';
                        document.getElementById('pointageId').value = item.id_pointage;
                        document.getElementById('datePointage').value = item.date_pointage;
                        document.getElementById('debutMois').value = item.kilometrage_debut_mois;
                        document.getElementById('finMois').value = item.kilometrage_fin_mois;
                        document.getElementById('nombreVoyages').value = item.nombre_voyages;
                        document.getElementById('kilometrageParVoyage').value = item.kilometrage_par_voyage_estime;
                        
                        loadVehiculesAndChauffeurs(item.id_vehicule, item.id_chauffeur);
                        pointageModal.classList.remove('hidden');
                    } else {
                        alert('Erreur lors de la récupération des données.');
                    }
                });
        }
        
        function loadVehiculesAndChauffeurs(selectedVehicule = null, selectedChauffeur = null) {
            fetch('php/api_pointage_vehicule.php?action=get_vehicules')
                .then(response => response.json())
                .then(data => {
                    vehiculeSelect.innerHTML = '<option value="">Sélectionner un véhicule</option>';
                    data.data.forEach(v => {
                        const option = document.createElement('option');
                        option.value = v.id_vehicule;
                        option.textContent = `${v.immatriculation} (${v.marque})`;
                        if (v.id_vehicule == selectedVehicule) { option.selected = true; }
                        vehiculeSelect.appendChild(option);
                    });
                });
            
            fetch('php/api_pointage_vehicule.php?action=get_chauffeurs')
                .then(response => response.json())
                .then(data => {
                    chauffeurSelect.innerHTML = '<option value="">Sélectionner un chauffeur</option>';
                    data.data.forEach(c => {
                        const option = document.createElement('option');
                        option.value = c.id_chauffeur;
                        option.textContent = c.nom_chauffeur;
                        if (c.id_chauffeur == selectedChauffeur) { option.selected = true; }
                        chauffeurSelect.appendChild(option);
                    });
                });
        }
        
        filterMonth.addEventListener('change', () => { currentPage = 1; fetchData(); });
        filterSearch.addEventListener('input', () => { currentPage = 1; fetchData(); });
        });
                // Charger la liste des clients
        // Charger la liste des clients
        function loadClients() {
            const select = document.getElementById('factureClient');
            select.innerHTML = '<option value="">Chargement...</option>';

            fetch('php/api_facturation.php?action=get_clients')
                .then(res => res.json())
                .then(data => {
                    if (!Array.isArray(data) || data.length === 0) {
                        select.innerHTML = '<option value="">Aucun client trouvé</option>';
                        return;
                    }

                    select.innerHTML = '<option value="">Sélectionner un client</option>';
                    data.forEach(c => {
                        select.innerHTML += `<option value="${c.id_client}">${c.nom_client}</option>`;
                    });
                })
                .catch(err => {
                    console.error('Erreur chargement clients:', err);
                    select.innerHTML = '<option value="">Erreur de chargement</option>';
                });
        }

        // Appel automatique dès que le DOM est prêt
        document.addEventListener('DOMContentLoaded', () => {
            loadClients();
        });



                // Charger véhicules du client (uniquement non facturés)
        let selectedClient = null;

        // Quand on change de client
        document.getElementById('factureClient').addEventListener('change', function () {
            selectedClient = this.value;
            document.querySelector('#factureVehiculesTable tbody').innerHTML = '';

            if (!selectedClient) return;

            // On ne charge pas encore les pointages tant que le mois n'est pas choisi
            if (document.getElementById('moisFacturation').value) {
                checkAndLoadPointages();
            }
        });

        // Quand on change de mois
        document.getElementById('moisFacturation').addEventListener('change', function () {
            if (!selectedClient) {
                alert("Veuillez d'abord sélectionner un client.");
                this.value = '';
                return;
            }
            checkAndLoadPointages();
        });

        // Fonction qui vérifie si le mois est déjà facturé et charge les pointages
        function checkAndLoadPointages() {
            const mois = document.getElementById('moisFacturation').value;
            const tbody = document.querySelector('#factureVehiculesTable tbody');
            tbody.innerHTML = '';

            if (!mois) return;

            // 1️⃣ Vérifier si le mois est déjà facturé pour ce client
            fetch(`php/api_facturation.php?action=check_month_factured&id_client=${selectedClient}&mois=${mois}`)
                .then(res => res.json())
                .then(data => {
                    if (data.factured) {
                        alert(`⚠ Le mois ${mois} est déjà facturé pour ce client. Veuillez choisir un autre mois.`);
                        document.getElementById('moisFacturation').value = '';
                        return;
                    }

                    // 2️⃣ Charger les pointages de ce mois pour ce client
                    fetch(`php/api_facturation.php?action=get_vehicules_client&id_client=${selectedClient}&mois=${mois}`)
                        .then(res => res.json())
                        .then(pointages => {
                            if (!Array.isArray(pointages) || pointages.length === 0) {
                                alert(`Aucun pointage trouvé pour ${mois} pour ce client.`);
                                return;
                            }

                            tbody.innerHTML = pointages.map(v => `
                                <tr data-id-pointage="${v.id_pointage}" data-km="${v.kilometrage_total}">
                                    <td>${v.immatriculation}</td>
                                    <td>${v.kilometrage_total} Km</td>
                                    <td class="montant_ht">0</td>
                                    <td class="montant_tva">0</td>
                                    <td class="montant_ttc">0</td>
                                </tr>
                            `).join('');

                            recalculerMontants();
                        })
                        .catch(err => {
                            console.error('Erreur chargement pointages:', err);
                            tbody.innerHTML = `<tr><td colspan="5" class="text-center text-red-500">Erreur de chargement</td></tr>`;
                        });
                })
                .catch(err => {
                    console.error('Erreur vérification mois facturé:', err);
                    alert('Erreur lors de la vérification du mois.');
                });
        }




        // Recalcul automatique des montants quand tarif ou TVA changent
        ['tarifKm', 'tva'].forEach(id => {
            document.getElementById(id).addEventListener('input', () => {
                if (document.querySelector('#factureVehiculesTable tbody tr[data-id-pointage]')) {
                    recalculerMontants();
                }
            });
        });



        function recalculerMontants() {
            const tarif = parseFloat(document.getElementById('tarifKm').value) || 0;
            const tva = parseFloat(document.getElementById('tva').value) || 0;

            document.querySelectorAll('#factureVehiculesTable tbody tr[data-id-pointage]').forEach(row => {
                const km = parseFloat(row.dataset.km) || 0;
                const montant_ht  = km * tarif;
                const montant_tva = montant_ht * (tva / 100);
                const montant_ttc = montant_ht + montant_tva;

                row.querySelector('.montant_ht').textContent  = montant_ht.toFixed(2);
                row.querySelector('.montant_tva').textContent = montant_tva.toFixed(2);
                row.querySelector('.montant_ttc').textContent = montant_ttc.toFixed(2);
            });
        }

        function loadHistoriqueFactures() {
            fetch('php/api_facturation.php?action=get_historique')
                .then(r => r.json())
                .then(data => {
                    const tbody = document.querySelector('#historiqueFacturesTable tbody');
                    tbody.innerHTML = '';

                    if (!data || data.length === 0) {
                        tbody.innerHTML = '<tr><td colspan="9" class="text-center">Aucune facture enregistrée</td></tr>';
                        return;
                    }

                    data.forEach(facture => {
                        const row = document.createElement('tr');
                        row.innerHTML = `
                            <td>${facture.numero_facture}</td>
                            <td>${facture.client}</td>
                            <td>${facture.mois_facture || ''}</td>
                            <td>${facture.date_facture ? new Date(facture.date_facture).toLocaleDateString('fr-FR') : ''}</td>
                            <td>${facture.total_km.toLocaleString()} Km</td>
                            <td>${facture.total_ht.toLocaleString('fr-FR', { style: 'currency', currency: 'MGA' })}</td>
                            <td>${facture.total_tva.toLocaleString('fr-FR', { style: 'currency', currency: 'MGA' })}</td>
                            <td>${facture.total_ttc.toLocaleString('fr-FR', { style: 'currency', currency: 'MGA' })}</td>
                            <td>
                                <button class="btn btn-info btn-sm" onclick="genererFacture('${facture.numero_facture}')">
                                    <i class="fas fa-file-pdf"></i> PDF
                                </button>
                            </td>
                        `;
                        tbody.appendChild(row);
                    });
                })
                .catch(err => {
                    console.error('Erreur chargement historique', err);
                });
        }


            // Soumission du formulaire
        document.getElementById('facturationForm').addEventListener('submit', function(e) {
            e.preventDefault();

            const id_client = document.getElementById('factureClient').value;
            const mois_facturation = document.getElementById('moisFacturation').value;
            const tarif_km = parseFloat(document.getElementById('tarifKm').value) || 0;
            const tva = parseFloat(document.getElementById('tva').value) || 0;

            fetch('php/api_facturation.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'save_factures',
                    id_client,
                    mois_facturation,
                    tarif_km,
                    tva
                })
            })
            .then(r => r.json())
            .then(data => {
                if (!data.success) {
                    alert("Erreur: " + data.error);
                    return;
                }
                // 1️⃣ Message utilisateur
                alert(data.message);

                // 2️⃣ Recharge l’historique immédiatement
                loadHistoriqueFactures();

                // 3️⃣ Vide la table de prévisualisation
                document.querySelector('#factureVehiculesTable tbody').innerHTML = '';

                // 4️⃣ Reset des champs si besoin
                document.getElementById('moisFacturation').value = '';
                document.getElementById('tarifKm').value = '';
                document.getElementById('tva').value = '';
            })
            .catch(err => {
                alert("Erreur réseau: " + err);
            });
        });


        

    </script>
    <script>
        // Fonction globale pour ouvrir la facture PDF générée par generate_facture_km.php
        window.genererFacture = function(numeroFacture) {
            // Construire l’URL vers ton script de génération
            const url = `generate_facture_km.php?numero_facture=${encodeURIComponent(numeroFacture)}`;

            // Ouvrir dans un nouvel onglet pour éviter de casser la page en cours
            window.open(url, '_blank', 'noopener');
        };
    </script>
</body>
</html>
