<?php
// ASE_Maintenance/admin.php

session_start();

// Pour le débogage :
// ini_set('display_errors', 1);
// error_reporting(E_ALL);

require_once 'php/config.php';

// Définit le fuseau horaire à Antananarivo, Madagascar
date_default_timezone_set('Indian/Antananarivo');

// Vérification de l'authentification et du rôle Administrateur
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true || $_SESSION['role'] !== 'Administrateur') {
    header('Location: index.php'); // Rediriger vers la page de connexion si non autorisé
    exit();
}

$pdo = getDbConnection();

$message = '';
$message_type = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

$records_per_page_details = 10; // Nombre d'enregistrements par page pour les modales de détails
$records_per_page_dashboard_table = 5; // Nombre d'enregistrements par page pour les tableaux du tableau de bord

// --- Nouveaux Endpoints AJAX pour les détails de stock paginés et tableaux du dashboard ---

// Endpoint pour obtenir toutes les pièces en stock avec pagination (pour modal)
if (isset($_GET['action']) && $_GET['action'] === 'get_all_pieces_in_stock') {
    header('Content-Type: application/json');
    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($current_page - 1) * $records_per_page_details;

    $stmt_total = $pdo->query("SELECT COUNT(*) FROM piece WHERE stock_initial > 0");
    $total_records = $stmt_total->fetchColumn() ?? 0;
    $total_pages = ceil($total_records / $records_per_page_details);

    $stmt_pieces = $pdo->prepare("
        SELECT
            p.id_piece,
            p.nom_piece,
            p.reference,
            p.stock_initial,
            p.prix_unitaire,
            p.tva,
            tp.nom_type,
            r.nom_rayon
        FROM
            piece p
        JOIN
            type_piece tp ON p.id_type = tp.id_type
        JOIN
            rayon r ON tp.id_rayon = r.id_rayon
        WHERE p.stock_initial > 0
        ORDER BY p.nom_piece ASC
        LIMIT ? OFFSET ?
    ");
    $stmt_pieces->execute([$records_per_page_details, $offset]);
    $pieces_data = $stmt_pieces->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true,
        'data' => $pieces_data,
        'current_page' => $current_page,
        'total_pages' => $total_pages,
        'total_records' => $total_records
    ]);
    exit();
}

// Endpoint pour obtenir tous les pneus en stock avec pagination (pour modal)
if (isset($_GET['action']) && $_GET['action'] === 'get_all_pneus_in_stock') {
    header('Content-Type: application/json');
    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    // $records_per_page_details est déjà défini plus haut (10)
    $offset = ($current_page - 1) * $records_per_page_details;

    // Count total unique pneu models with stock > 0
    $stmt_total = $pdo->query("
        SELECT COUNT(*)
        FROM pneus p
        WHERE p.quantiter_initial > 0
    ");
    $total_records = $stmt_total->fetchColumn() ?? 0;
    $total_pages = ceil($total_records / $records_per_page_details);

    // Get pneu models with their aggregated stock
    $stmt_pneus = $pdo->prepare("
        SELECT
            p.id_pneu,
            p.nom_modele,
            m.nom_marque,
            d.valeur_dimension,
            pr.nom_profil,
            p.prix_achat_ht,
            p.tva_applicable,
            p.quantiter_initial AS quantite_en_stock
        FROM
            pneus p
        JOIN
            marques_pneu m ON p.id_marque = m.id_marque
        JOIN
            dimensions_pneu d ON p.id_dimension = d.id_dimension
        JOIN
            profils_pneu pr ON p.id_profil = pr.id_profil
        WHERE p.quantiter_initial > 0
        ORDER BY p.nom_modele ASC
        LIMIT ? OFFSET ?
    ");
    $stmt_pneus->execute([$records_per_page_details, $offset]);
    $pneus_data = $stmt_pneus->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true,
        'data' => $pneus_data,
        'current_page' => $current_page,
        'total_pages' => $total_pages,
        'total_records' => $total_records
    ]);
    exit();
}

// Endpoint pour les Dernières Sorties de Pièces (pour tableau du dashboard)
if (isset($_GET['action']) && $_GET['action'] === 'get_sorties_pieces_data') {
    header('Content-Type: application/json');
    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($current_page - 1) * $records_per_page_dashboard_table;

    $stmt_total = $pdo->query("SELECT COUNT(*) FROM sorties_pieces");
    $total_records = $stmt_total->fetchColumn() ?? 0;
    $total_pages = ceil($total_records / $records_per_page_dashboard_table);

    $stmt_data = $pdo->prepare("
        SELECT
            sp.id_sortie,
            sp.date_sortie,
            u.nom_utilisateur AS magasinier,
            i.id_intervention,
            v.immatriculation AS vehicule_immat,
            sp.total_ttc
        FROM
            sorties_pieces sp
        JOIN
            utilisateurs u ON sp.id_magasinier = u.id_utilisateur
        JOIN
            interventions i ON sp.id_intervention = i.id_intervention
        JOIN
            vehicules v ON i.id_vehicule = v.id_vehicule
        ORDER BY sp.date_sortie DESC
        LIMIT ? OFFSET ?
    ");
    $stmt_data->execute([$records_per_page_dashboard_table, $offset]);
    $data = $stmt_data->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true,
        'data' => $data,
        'current_page' => $current_page,
        'total_pages' => $total_pages,
        'total_records' => $total_records
    ]);
    exit();
}

// Endpoint pour les Dernières Sorties de Pneus (pour tableau du dashboard)
if (isset($_GET['action']) && $_GET['action'] === 'get_sorties_pneus_data') {
    header('Content-Type: application/json');
    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($current_page - 1) * $records_per_page_dashboard_table;

    $stmt_total = $pdo->query("
        SELECT COUNT(*)
        FROM sorties_pneu sp
        JOIN details_sortie_pneu dsp ON sp.id_sortie_pneu = dsp.id_sortie_pneu
        JOIN inventaire_pneu ip ON dsp.id_inventaire_pneu = ip.id_inventaire_pneu
    ");
    $total_records = $stmt_total->fetchColumn() ?? 0;
    $total_pages = ceil($total_records / $records_per_page_dashboard_table);

    $stmt_data = $pdo->prepare("
        SELECT
            sp.id_sortie_pneu,
            sp.date_sortie,
            u.nom_utilisateur AS magasinier,
            ip.numero_serie,
            p.nom_modele,
            mpv.position_pneu,
            v.immatriculation,
            ip.statut_pneu AS statut_final,
            sp.raison_sortie
        FROM
            sorties_pneu sp
        JOIN
            utilisateurs u ON sp.id_magasinier = u.id_utilisateur
        JOIN
            details_sortie_pneu dsp ON sp.id_sortie_pneu = dsp.id_sortie_pneu
        JOIN
            inventaire_pneu ip ON dsp.id_inventaire_pneu = ip.id_inventaire_pneu
        JOIN
            pneus p ON ip.id_pneu = p.id_pneu
        LEFT JOIN
            montage_pneu_vehicule mpv ON ip.id_inventaire_pneu = mpv.id_inventaire_pneu
            AND mpv.date_montage <= sp.date_sortie
            AND (mpv.date_demontage IS NULL OR mpv.date_demontage > sp.date_sortie)
        LEFT JOIN
            vehicules v ON mpv.id_vehicule = v.id_vehicule
        ORDER BY sp.date_sortie DESC
        LIMIT ? OFFSET ?
    ");
    $stmt_data->execute([$records_per_page_dashboard_table, $offset]);
    $data = $stmt_data->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true,
        'data' => $data,
        'current_page' => $current_page,
        'total_pages' => $total_pages,
        'total_records' => $total_records
    ]);
    exit();
}

// Endpoint pour les Alertes Stock Bas (Pièces) (pour tableau du dashboard)
if (isset($_GET['action']) && $_GET['action'] === 'get_alertes_pieces_data') {
    header('Content-Type: application/json');
    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($current_page - 1) * $records_per_page_dashboard_table;

    $stmt_total = $pdo->query("SELECT COUNT(*) FROM piece p JOIN type_piece tp ON p.id_type = tp.id_type WHERE p.stock_initial <= p.seuil_alerte AND p.stock_initial > 0");
    $total_records = $stmt_total->fetchColumn() ?? 0;
    $total_pages = ceil($total_records / $records_per_page_dashboard_table);

    $stmt_data = $pdo->prepare("
        SELECT
            p.nom_piece,
            p.reference,
            p.stock_initial,
            p.seuil_alerte,
            tp.nom_type
        FROM
            piece p
        JOIN
            type_piece tp ON p.id_type = tp.id_type
        WHERE
            p.stock_initial <= p.seuil_alerte AND p.stock_initial > 0
        ORDER BY p.stock_initial ASC
        LIMIT ? OFFSET ?
    ");
    $stmt_data->execute([$records_per_page_dashboard_table, $offset]);
    $data = $stmt_data->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true,
        'data' => $data,
        'current_page' => $current_page,
        'total_pages' => $total_pages,
        'total_records' => $total_records
    ]);
    exit();
}

// Endpoint pour les Alertes Stock Bas (Pneus) (pour tableau du dashboard)
if (isset($_GET['action']) && $_GET['action'] === 'get_alertes_pneus_data') {
    header('Content-Type: application/json');
    $current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($current_page - 1) * $records_per_page_dashboard_table;
    $alert_threshold_pneus = 5; // Seuil d'alerte des pneus

    $stmt_total = $pdo->prepare("
        SELECT COUNT(*)
        FROM pneus p
        WHERE p.quantiter_initial <= ? AND p.quantiter_initial > 0
    ");
    $stmt_total->execute([$alert_threshold_pneus]);
    $total_records = $stmt_total->fetchColumn() ?? 0;
    $total_pages = ceil($total_records / $records_per_page_dashboard_table);

    $stmt_data = $pdo->prepare("
        SELECT
            p.nom_modele,
            m.nom_marque,
            d.valeur_dimension,
            pr.nom_profil,
            p.quantiter_initial
        FROM
            pneus p
        JOIN marques_pneu m ON p.id_marque = m.id_marque
        JOIN dimensions_pneu d ON p.id_dimension = d.id_dimension
        JOIN profils_pneu pr ON p.id_profil = pr.id_profil
        WHERE
            p.quantiter_initial <= ? AND p.quantiter_initial > 0
        ORDER BY p.quantiter_initial ASC
        LIMIT ? OFFSET ?
    ");
    $stmt_data->execute([$alert_threshold_pneus, $records_per_page_dashboard_table, $offset]);
    $data = $stmt_data->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true,
        'data' => $data,
        'current_page' => $current_page,
        'total_pages' => $total_pages,
        'total_records' => $total_records
    ]);
    exit();
}


// --- Récupération des Indicateurs Clés (KPIs) ---

// 1a. CA Global de toutes les interventions (Pièces)
$stmt_ca_pieces = $pdo->query("SELECT SUM(dsp.prix_total_ttc_ligne) AS ca_global_pieces FROM details_sortie_piece dsp");
$ca_global_pieces = $stmt_ca_pieces->fetchColumn() ?? 0;

// 1c. CA des Sorties de Pneus (Valeur des Pneus Sortis)
// Cette requête agrège la valeur TTC des pneus sortis via les enregistrements de sorties_pneu.
$stmt_ca_pneus = $pdo->query("
    SELECT SUM(p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS valeur_totale_pneus_sortis_ttc
    FROM sorties_pneu sp
    JOIN details_sortie_pneu dsp ON sp.id_sortie_pneu = dsp.id_sortie_pneu
    JOIN inventaire_pneu ip ON dsp.id_inventaire_pneu = ip.id_inventaire_pneu
    JOIN pneus p ON ip.id_pneu = p.id_pneu;
");
$ca_pneus_sortis = $stmt_ca_pneus->fetchColumn() ?? 0;


// 2a. Coût des Réceptions de Pièces
$stmt_cout_receptions_pieces = $pdo->query("
    SELECT
        SUM(lr.quantite_receptionnee * lr.prix_achat_ht_reception * (1 + lr.tva_reception / 100)) AS cout_total_receptions_pieces_ttc
    FROM
        lignes_reception lr
");
$cout_receptions_pieces = $stmt_cout_receptions_pieces->fetchColumn() ?? 0;

// 2b. Coût des Réceptions de Pneus
$stmt_cout_receptions_pneus = $pdo->query("
    SELECT
        SUM(lrp.quantite_receptionnee * p.prix_achat_ht * (1 + p.tva_applicable / 100)) AS cout_total_receptions_pneus_ttc
    FROM
        lignes_reception_pneu lrp
    JOIN
        pneus p ON lrp.id_pneu = p.id_pneu
");
$cout_receptions_pneus = $stmt_cout_receptions_pneus->fetchColumn() ?? 0;

// Nombre total d'interventions terminées
$stmt_total_interventions = $pdo->query("SELECT COUNT(*) FROM interventions WHERE statut = 'Terminee'");
$total_interventions = $stmt_total_interventions->fetchColumn() ?? 0;

// Nombre total de pneus "Montés"
$stmt_pneus_montes_count = $pdo->query("SELECT COUNT(*) FROM inventaire_pneu WHERE statut_pneu = 'Monté'");
$pneus_montes_count = $stmt_pneus_montes_count->fetchColumn() ?? 0;

// Nombre total de pièces en stock
$stmt_pieces_en_stock_count = $pdo->query("SELECT COUNT(*) FROM piece WHERE stock_initial > 0"); // Compter les pièces avec stock > 0
$pieces_en_stock_count = $stmt_pieces_en_stock_count->fetchColumn() ?? 0;

// Nombre total de pneus en stock (compté par numéro de série unique)
$stmt_pneus_en_stock_count = $pdo->query("SELECT COUNT(*) FROM inventaire_pneu WHERE statut_pneu = 'En stock'");
$pneus_en_stock_count = $stmt_pneus_en_stock_count->fetchColumn() ?? 0;


// --- Données pour le graphique (exemple: CA des pièces par mois) ---
$stmt_ca_mensuel_pieces = $pdo->query("
    SELECT
        DATE_FORMAT(sp.date_sortie, '%Y-%m') AS mois,
        SUM(dsp.prix_total_ttc_ligne) AS ca_mensuel
    FROM
        sorties_pieces sp
    JOIN
        details_sortie_piece dsp ON sp.id_sortie = dsp.id_sortie
    GROUP BY
        mois
    ORDER BY
        mois ASC
    LIMIT 12 -- Les 12 derniers mois
");
$ca_mensuel_pieces_data = $stmt_ca_mensuel_pieces->fetchAll(PDO::FETCH_ASSOC);

$chart_labels = [];
$chart_data = [];
foreach ($ca_mensuel_pieces_data as $row) {
    $chart_labels[] = $row['mois'];
    $chart_data[] = $row['ca_mensuel'];
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard Admin - ASE Maintenance</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <!-- Chart.js pour les graphiques -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <style>
        /* Variables CSS pour les couleurs du thème sombre */
        :root {
            --primary-bg: #010d18;
            --secondary-bg: #1a1a1a;
            --navbar-bg: rgba(1, 13, 24, 0.95);
            --text-color: #e0e0e0;
            --heading-color: #ffda47;
            --accent-color: #ffda47;
            --border-color: rgba(224, 224, 224, 0.1);
            --form-bg: #2b2b2b;
            --input-bg: #3c3c3c;
            --input-border: #505050;
            --table-header-bg: #101010;
            --table-row-even-bg: #282828;

            --btn-primary-bg: #007bff;
            --btn-primary-hover: #0056b3;
            --btn-warning-bg: #ffc107;
            --btn-warning-hover: #e0a800;
            --btn-danger-bg: #dc3545;
            --btn-danger-hover: #c82333;
            --btn-success-bg: #28a745;
            --btn-success-hover: #218838;
            --btn-info-bg: #17a2b8;
            --btn-info-hover: #138496;

            --success-bg: rgba(40, 167, 69, 0.2);
            --success-text: #28a745;
            --error-bg: rgba(220, 53, 69, 0.2);
            --error-text: #dc3545;
            --warning-bg: rgba(255, 193, 7, 0.2);
            --warning-text: #ffc107;

            --kpi-card-bg: #333333;
            --kpi-card-border: #505050;
            --kpi-text-color: #f8f8f8;
            --kpi-value-color: #ffda47;
        }

        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            overflow-x: hidden;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background-color: var(--primary-bg);
            color: var(--text-color);
            font-size: 16px;
        }

        .sparkles-container {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            pointer-events: none;
            overflow: hidden;
            z-index: -1;
            background-color: var(--primary-bg);
        }

        .spark {
            position: absolute;
            background-color: rgba(255, 255, 255, 0.7);
            border-radius: 50%;
            opacity: 0;
            animation: rain-sparkles 2s linear infinite, twinkle 1s ease-in-out infinite alternate;
        }

        @keyframes rain-sparkles {
            0% { transform: translateY(-10px) translateX(var(--random-x, 0)); opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { transform: translateY(calc(100vh + 10px)) translateX(var(--random-x, 0)); opacity: 0; }
        }

        @keyframes twinkle {
            from { opacity: 0.7; transform: scale(1); }
            to { opacity: 1; transform: scale(1.2); }
        }

        .navbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: var(--navbar-bg);
            padding: 15px 30px;
            color: white;
            position: sticky;
            top: 0;
            z-index: 10;
            width: 100%;
            box-sizing: border-box;
            border-bottom: 1px solid var(--border-color);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.5);
        }

        .navbar-brand {
            display: flex;
            align-items: center;
        }

        .navbar-brand img {
            height: 45px;
            margin-right: 15px;
            border-radius: 0;
            padding: 2px;
        }

        .navbar-brand span {
            font-size: 1.6em;
            font-weight: bold;
            color: white;
        }

        .navbar-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
        }

        .navbar-nav li {
            margin-left: 30px;
            position: relative; /* Pour le dropdown */
            font-size: 0.7em; /* Taille de police réduite */
        }

        .navbar-nav a {
            color: var(--text-color);
            text-decoration: none;
            padding: 8px 0;
            transition: color 0.3s ease, border-bottom 0.3s ease;
            position: relative;
            display: flex;
            align-items: center;
            font-size: 1.05em; /* Maintenir la taille pour le texte du lien */
        }

        .navbar-nav a i {
            margin-right: 10px;
            font-size: 1.1em; /* Maintenir la taille pour les icônes de lien */
        }
        
        /* Style spécifique pour l'icône de déconnexion */
        .navbar-nav .fa-power-off {
            font-size: 1.5em !important; /* Augmenter la taille pour l'icône de déconnexion */
        }

        .navbar-nav a:hover {
            color: var(--accent-color);
        }

        .navbar-nav a.active {
            color: var(--accent-color);
            font-weight: bold;
        }

        .navbar-nav a.active::after {
            content: '';
            position: absolute;
            left: 0;
            bottom: -5px;
            width: 100%;
            height: 3px;
            background-color: var(--accent-color);
            border-radius: 2px;
        }

        /* Styles pour le menu déroulant (dropdown) */
        .dropdown-menu {
            display: none; /* Caché par défaut, sera géré par JS */
            position: absolute;
            background-color: var(--navbar-bg);
            min-width: 200px;
            box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.6);
            z-index: 1;
            padding: 10px 0;
            border-radius: 8px;
            top: calc(100% + 5px); /* Position sous l'élément parent */
            left: 0;
            list-style: none;
        }

        .dropdown-menu li {
            margin: 0;
        }

        .dropdown-menu a {
            color: var(--text-color);
            padding: 12px 16px;
            text-decoration: none;
            display: block;
            text-align: left;
            font-size: 0.95em;
        }

        .dropdown-menu a:hover {
            background-color: rgba(255, 218, 71, 0.1);
            color: var(--accent-color);
        }

        .dropdown.active .dropdown-menu {
            display: block;
        }


        .burger-menu {
            display: none;
            background: none;
            border: none;
            cursor: pointer;
            padding: 10px;
            z-index: 11;
        }

        .burger-menu .bar {
            width: 30px;
            height: 3px;
            background-color: white;
            margin: 6px 0;
            transition: transform 0.3s ease-in-out, opacity 0.3s ease-in-out;
        }

        .burger-menu.open .bar:nth-child(1) { transform: rotate(-45deg) translate(-5px, 6px); }
        .burger-menu.open .bar:nth-child(2) { opacity: 0; }
        .burger-menu.open .bar:nth-child(3) { transform: rotate(45deg) translate(-5px, -6px); }

        .navbar-nav.open {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: absolute;
            top: 100%;
            left: 0;
            background-color: var(--navbar-bg);
            width: 100%;
            padding: 25px 0;
            box-shadow: 0 10px 20px rgba(0,0,0,0.3);
            z-index: 9;
        }

        .navbar-nav.open li {
            margin: 12px 0;
        }

        .main-content {
            flex-grow: 1;
            padding: 40px;
            background-color: var(--secondary-bg);
            color: var(--text-color);
            max-width: 1400px;
            width: 95%;
            margin: 30px auto;
            border-radius: 10px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.6);
        }

        h1 {
            color: var(--heading-color);
            text-align: center;
            margin-bottom: 40px;
            font-size: 3em;
            border-bottom: 4px solid var(--accent-color);
            padding-bottom: 15px;
            letter-spacing: 1px;
        }

        h2.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 12px;
            margin-top: 45px;
            margin-bottom: 25px;
            font-size: 2.2em;
        }
        h3.section-title {
            color: var(--heading-color);
            border-bottom: 2px solid var(--accent-color);
            padding-bottom: 8px;
            margin-top: 30px;
            margin-bottom: 20px;
            font-size: 1.8em;
        }

        .message {
            padding: 18px;
            margin-bottom: 30px;
            border-radius: 6px;
            font-weight: bold;
            font-size: 1.15em;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.3);
        }

        .message.success { background-color: var(--success-bg); color: var(--success-text); border: 1px solid var(--success-text); }
        .message.error { background-color: var(--error-bg); color: var(--error-text); border: 1px solid var(--error-text); }
        .message.warning { background-color: var(--warning-bg); color: var(--warning-text); border: 1px solid var(--warning-text); }

        .dashboard-section {
            background-color: var(--form-bg);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.4);
            margin-bottom: 40px;
            border: none;
        }

        /* KPI Cards */
        .kpi-cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .kpi-card {
            background-color: var(--kpi-card-bg);
            border: 1px solid var(--kpi-card-border);
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 2px 8px rgba(0,0,0,0.4);
            transition: transform 0.2s ease, background-color 0.2s ease;
            cursor: pointer; /* Added cursor for clickable cards */
        }
        .kpi-card:hover {
            transform: translateY(-5px);
            background-color: #4a4a4a; /* Slightly lighter on hover */
        }

        .kpi-card .icon {
            font-size: 2.5em;
            color: var(--accent-color);
            margin-bottom: 10px;
        }

        .kpi-card .label {
            font-size: 0.9em;
            color: var(--text-color);
            margin-bottom: 5px;
        }

        .kpi-card .value {
            font-size: 1.8em;
            font-weight: bold;
            color: var(--kpi-value-color);
        }

        /* Chart Styles */
        .chart-container {
            position: relative;
            height: 400px; /* Fixed height for consistency */
            width: 100%;
            background-color: var(--kpi-card-bg);
            padding: 20px;
            border-radius: 8px;
            border: 1px solid var(--kpi-card-border);
            box-shadow: 0 2px 8px rgba(0,0,0,0.4);
        }

        /* Table Styles (reused from previous pages) */
        .table-responsive {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            margin-top: 20px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.4);
            border: 1px solid var(--border-color);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background-color: var(--form-bg);
            min-width: 600px;
        }

        th, td {
            border: 1px solid var(--input-border);
            padding: 12px;
            text-align: left;
            font-size: 0.7em;
            color: var(--text-color);
            vertical-align: middle;
        }

        th {
            background-color: var(--table-header-bg);
            color: white;
            text-transform: uppercase;
            font-size: 0.8em;
            font-weight: bold;
        }

        tr:nth-child(even) {
            background-color: var(--table-row-even-bg);
        }
        tr:hover {
            background-color: #3a3a3a;
            transition: background-color 0.2s ease;
        }

        .low-stock-row {
            background-color: rgba(220, 53, 69, 0.3) !important; /* Rouge clair */
            font-weight: bold;
        }
        .low-stock-row td {
            color: var(--error-text);
        }

        /* Responsive */
        @media (max-width: 1024px) {
            .main-content {
                max-width: 95%;
                padding: 30px;
            }
            h1 { font-size: 2.5em; }
            h2.section-title, h3.section-title { font-size: 2em; }
            .kpi-cards-grid {
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            }
        }

        @media (max-width: 768px) {
            .navbar-nav { display: none; }
            .burger-menu { display: block; }
            .main-content {
                margin: 15px auto;
                padding: 20px;
                width: auto;
            }
            h1 { font-size: 2.2em; margin-bottom: 25px; }
            h2.section-title, h3.section-title { font-size: 1.8em; margin-top: 30px; }
            th, td { padding: 8px; font-size: 0.85em; }
            table { min-width: unset; }
            .kpi-cards-grid {
                grid-template-columns: 1fr; /* Stack on mobile */
            }
            /* Modal-specific responsive adjustments */
            .modal1-content {
                max-width: 95%; /* Adjust max-width for smaller screens */
            }
        }

        @media (max-width: 480px) {
            .main-content { padding: 15px; margin: 10px auto; }
            h1 { font-size: 1.8em; margin-bottom: 20px; }
            h2.section-title, h3.section-title { font-size: 1.5em; }
            .navbar { padding: 10px 15px; }
            .navbar-brand img { height: 35px; }
            .navbar-brand span { font-size: 1.2em; }
        }

        /* --- MODAL STYLES (copied from reception_pneu.php for consistency) --- */
        .modal1 {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.7);
            justify-content: center;
            align-items: center;
            -webkit-animation: fadeIn 0.3s forwards;
            animation: fadeIn 0.3s forwards;
        }

        .modal1-content {
            background-color: var(--form-bg);
            margin: auto;
            padding: 25px 30px;
            border: 1px solid var(--input-border);
            border-radius: 10px;
            width: 90%;
            max-width: 900px; /* Ajusté pour les tables de détails */
            box-shadow: 0 5px 15px rgba(0,0,0,0.5);
            position: relative;
            transform: scale(0.95);
            transition: transform 0.2s ease-out;
            -webkit-animation: slideIn 0.3s forwards;
            animation: slideIn 0.3s forwards;
        }

        .modal1.open {
            display: flex;
        }

        .modal1.open .modal1-content {
            transform: scale(1);
        }

        /* Modified from .close-button to .close1-button */
        .close1-button {
            color: var(--text-color);
            position: absolute;
            top: 15px;
            right: 25px;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            transition: color 0.3s ease;
        }

        .close1-button:hover,
        .close1-button:focus {
            color: var(--accent-color);
            text-decoration: none;
            cursor: pointer;
        }

        @-webkit-keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @keyframes fadeIn { from {opacity: 0;} to {opacity: 1;} }
        @-webkit-keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @keyframes slideIn { from {transform: translateY(-50px) scale(0.95);} to {transform: translateY(0) scale(1);} }
        @-webkit-keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @keyframes fadeOut { from {opacity: 1;} to {opacity: 0;} }
        @-webkit-keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        @keyframes slideOut { from {transform: translateY(0) scale(1);} to {transform: translateY(-50px) scale(0.95);} }
        /* End MODAL STYLES */

        /* Styles for dynamic pagination (re-applied and enhanced) */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 20px;
            gap: 8px; /* Slightly reduced gap */
        }

        .pagination a,
        .pagination span {
            background-color: var(--input-bg);
            color: var(--text-color);
            padding: 8px 14px; /* Adjusted padding */
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease, color 0.3s ease, transform 0.2s ease;
            font-weight: 500; /* Slightly less bold */
            font-size: 0.9em; /* Slightly smaller font */
            border: 1px solid var(--input-border); /* Add a subtle border */
        }

        .pagination a:hover {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            transform: translateY(-2px);
            box-shadow: 0 2px 8px rgba(255, 218, 71, 0.3);
        }

        .pagination span.current-page {
            background-color: var(--accent-color);
            color: var(--primary-bg);
            cursor: default;
            font-weight: 700; /* Bold for active page */
            box-shadow: 0 2px 8px rgba(255, 218, 71, 0.3);
        }

        .pagination span.disabled {
            opacity: 0.6;
            cursor: not-allowed;
            background-color: var(--table-row-even-bg); /* Muted background for disabled */
            color: #888;
        }

    </style>
</head>
<body>
    <nav class="navbar">
        <div class="navbar-brand">
            <img src="img/logo_ase.png" alt="Logo Entreprise">
        </div>
        <button class="burger-menu" aria-label="Ouvrir/Fermer le menu">
            <div class="bar"></div>
            <div class="bar"></div>
            <div class="bar"></div>
        </button>
        <ul class="navbar-nav">
            <li><a href="admin"><i class="fas fa-tachometer-alt"></i> Dashboard Admin</a></li>
            <li><a href="gestion_utilisateurs"><i class="fas fa-users-cog"></i> Gestion Utilisateurs</a></li>
            <li class="dropdown active">
                <a href="#" class="dropdown-toggle" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-chart-line"></i> Rapports & Analyses -- <i class="fas fa-caret-down"></i>
                </a>
                <ul class="dropdown-menu">
                    <li><a href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports Généraux</a></li>
                    <li><a href="details_sorties_globales" class="active"><i class="fas fa-file-invoice-dollar"></i> Sorties Détaillées (TTC)</a></li>
                    <li><a href="carnet_entretien_vehicule"><i class="fas fa-book"></i> Carnet d'Entretien Véhicule</a></li>
                </ul>
            </li>
            <li><a href="rapport_pneumatique"><i class="fas fa-history"></i> Rapport pneumatique</a></li>
            <li><a href="pointage_vehicule"><i class="fas fa-car"></i> Pointage Véhicules</a></li>
            <li><a href="php/authentification.php?action=logout"> <i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>

    <div class="sparkles-container"></div>

    <div class="main-content">
        <h1 id="page-title">Tableau de Bord Administrateur</h1>

        <?php if ($message): ?>
            <div class="message <?php echo htmlspecialchars($message_type); ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <!-- Section des Indicateurs Clés (KPIs) -->
        <div class="dashboard-section">
            <h2 class="section-title">Indicateurs Clés</h2>
            <div class="kpi-cards-grid">
                <div class="kpi-card">
                    <div class="icon"><i class="fas fa-dollar-sign"></i></div>
                    <div class="label">CA total sortie Pièces (TTC)</div>
                    <div class="value"><?php echo number_format($ca_global_pieces, 2, ',', ' '); ?> Ar</div>
                </div>
                <div class="kpi-card">
                    <div class="icon"><i class="fas fa-shipping-fast"></i></div>
                    <div class="label">Coût entrée Pièces (TTC)</div>
                    <div class="value"><?php echo number_format($cout_receptions_pieces, 2, ',', ' '); ?> Ar</div>
                </div>
                <div class="kpi-card">
                    <div class="icon"><i class="fas fa-road"></i></div> <!-- Changed icon to fa-road as requested -->
                    <div class="label">Valeur Pneus Sortis (TTC)</div>
                    <div class="value"><?php echo number_format($ca_pneus_sortis, 2, ',', ' '); ?> Ar</div>
                </div>
                <div class="kpi-card">
                    <div class="icon"><i class="fas fa-truck-ramp-box"></i></div>
                    <div class="label">Coût entrée Pneus (TTC)</div>
                    <div class="value"><?php echo number_format($cout_receptions_pneus, 2, ',', ' '); ?> Ar</div>
                </div>
                <div class="kpi-card">
                    <div class="icon"><i class="fas fa-tools"></i></div>
                    <div class="label">Interventions Terminées</div>
                    <div class="value"><?php echo number_format($total_interventions, 0, ',', ' '); ?></div>
                </div>
                <div class="kpi-card">
                    <div class="icon"><i class="fas fa-car-side"></i></div>
                    <div class="label">Pneus Montés Actuellement</div>
                    <div class="value"><?php echo number_format($pneus_montes_count, 0, ',', ' '); ?></div>
                </div>
                <div class="kpi-card" id="kpi-pieces-stock">
                    <div class="icon"><i class="fas fa-boxes"></i></div>
                    <div class="label">Pièces en Stock</div>
                    <div class="value"><?php echo number_format($pieces_en_stock_count, 0, ',', ' '); ?></div>
                </div>
                <div class="kpi-card" id="kpi-pneus-stock">
                    <div class="icon"><i class="fas fa-warehouse"></i></div>
                    <div class="label">Pneus en Stock</div>
                    <div class="value"><?php echo number_format($pneus_en_stock_count, 0, ',', ' '); ?></div>
                </div>
            </div>
        </div>

        <!-- Section Graphique (Exemple : Évolution du CA Pièces) -->
        <div class="dashboard-section">
            <h2 class="section-title">Évolution du CA Pièces (TTC par Mois)</h2>
            <div class="chart-container">
                <canvas id="caPiecesChart"></canvas>
            </div>
        </div>

        <!-- Section des Dernières Sorties de Pièces -->
        <div class="dashboard-section">
            <h2 class="section-title">Dernières Sorties de Pièces</h2>
            <div class="table-responsive">
                <table id="sortiesPiecesTable">
                    <thead>
                        <tr>
                            <th>ID Sortie</th>
                            <th>Date Sortie</th>
                            <th>Magasinier</th>
                            <th>ID Interv.</th>
                            <th>Véhicule</th>
                            <th>Total TTC</th>
                        </tr>
                    </thead>
                    <tbody id="sortiesPiecesTableBody">
                        <!-- Data will be loaded via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="sortiesPiecesPagination">
                <!-- Pagination will be generated here -->
            </div>
        </div>

        <!-- Section des Dernières Sorties de Pneus -->
        <div class="dashboard-section">
            <h2 class="section-title">Dernières Sorties de Pneus</h2>
            <div class="table-responsive">
                <table id="sortiesPneusTable">
                    <thead>
                        <tr>
                            <th>ID Sortie</th>
                            <th>Date Sortie</th>
                            <th>Magasinier</th>
                            <th>N° Série Pneu</th>
                            <th>Modèle</th>
                            <th>Position (Véhicule)</th>
                            <th>Statut Final</th>
                            <th>Raison</th>
                        </tr>
                    </thead>
                    <tbody id="sortiesPneusTableBody">
                        <!-- Data will be loaded via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="sortiesPneusPagination">
                <!-- Pagination will be generated here -->
            </div>
        </div>

        <!-- Section des Alertes Stock Bas (Pièces) -->
        <div class="dashboard-section">
            <h2 class="section-title">Alertes Stock Bas (Pièces)</h2>
            <div class="table-responsive">
                <table id="alertesPiecesTable">
                    <thead>
                        <tr>
                            <th>Nom Pièce</th>
                            <th>Référence</th>
                            <th>Type</th>
                            <th>Stock Actuel</th>
                            <th>Seuil Alerte</th>
                        </tr>
                    </thead>
                    <tbody id="alertesPiecesTableBody">
                        <!-- Data will be loaded via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="alertesPiecesPagination">
                <!-- Pagination will be generated here -->
            </div>
        </div>

        <!-- Section des Alertes Stock Bas (Pneus) -->
        <div class="dashboard-section">
            <h2 class="section-title">Alertes Stock Bas (Pneus)</h2>
            <div class="table-responsive">
                <table id="alertesPneusTable">
                    <thead>
                        <tr>
                            <th>Modèle</th>
                            <th>Marque</th>
                            <th>Dimension</th>
                            <th>Profil</th>
                            <th>Quantité en Stock</th>
                        </tr>
                    </thead>
                    <tbody id="alertesPneusTableBody">
                        <!-- Data will be loaded via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="alertesPneusPagination">
                <!-- Pagination will be generated here -->
            </div>
        </div>

    </div>

    <!-- Modal pour les détails des Pièces en Stock -->
    <div id="piecesInStockModal" class="modal1">
        <div class="modal1-content">
            <span class="close1-button" id="closePiecesInStockModal">&times;</span>
            <h2 class="section-title">Détails des Pièces en Stock</h2>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID Pièce</th>
                            <th>Nom Pièce</th>
                            <th>Référence</th>
                            <th>Type</th>
                            <th>Rayon</th>
                            <th>Stock Actuel</th>
                            <th>Prix Unitaire HT</th>
                        </tr>
                    </thead>
                    <tbody id="piecesInStockTableBody">
                        <!-- Les données seront chargées ici via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="piecesInStockPagination">
                <!-- Pagination sera générée ici -->
            </div>
        </div>
    </div>

    <!-- Modal pour les détails des Pneus en Stock -->
    <div id="pneusInStockModal" class="modal1">
        <div class="modal1-content">
            <span class="close1-button" id="closePneusInStockModal">&times;</span>
            <h2 class="section-title">Détails des Pneus en Stock</h2>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID Modèle</th>
                            <th>Modèle</th>
                            <th>Marque</th>
                            <th>Dimension</th>
                            <th>Profil</th>
                            <th>Quantité en Stock</th>
                            <th>Prix Achat HT</th>
                            <th>TVA</th>
                        </tr>
                    </thead>
                    <tbody id="pneusInStockTableBody">
                        <!-- Les données seront chargées ici via AJAX -->
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="pneusInStockPagination">
                <!-- Pagination sera générée ici -->
            </div>
        </div>
    </div>

<?php include('message.php'); ?>
    <script>
        // Fonction globale pour afficher les messages (déjà définie dans d'autres pages)
        function showMessage(msg, type) {
            const mainContent = document.querySelector('.main-content');
            let messageDiv = document.querySelector('.dynamic-message');

            if (messageDiv) {
                messageDiv.remove();
            }

            messageDiv = document.createElement('div');
            messageDiv.classList.add('message', 'dynamic-message', type);
            messageDiv.innerHTML = `<i class="fas ${type === 'success' ? 'fa-check-circle' : (type === 'error' ? 'fa-times-circle' : 'fa-info-circle')}"></i> ${msg}`;

            const pageTitle = document.getElementById('page-title');
            if (pageTitle && mainContent) {
                 mainContent.insertBefore(messageDiv, pageTitle.nextSibling);
            } else if (mainContent) {
                mainContent.prepend(messageDiv);
            }

            setTimeout(() => {
                if (messageDiv) {
                    messageDiv.style.opacity = '0';
                    messageDiv.style.transition = 'opacity 0.5s ease-out';
                    setTimeout(() => messageDiv.remove(), 500);
                }
            }, 5000);
        }

        // --- Fonctions pour charger et afficher les détails de stock dans les modales (AJAX) ---
        // Rendre ces fonctions globales pour être appelées par la pagination générique
        async function loadPiecesStockDetails(page = 1) {
            try {
                const response = await fetch(`admin.php?action=get_all_pieces_in_stock&page=${page}`);
                const data = await response.json();

                if (data.success) {
                    const piecesInStockTableBody = document.getElementById('piecesInStockTableBody');
                    const piecesInStockPagination = document.getElementById('piecesInStockPagination');
                    piecesInStockTableBody.innerHTML = ''; // Clear existing rows
                    if (data.data.length === 0) {
                        piecesInStockTableBody.innerHTML = '<tr><td colspan="7" class="text-center">Aucune pièce en stock.</td></tr>';
                    } else {
                        data.data.forEach(piece => {
                            const row = `
                                <tr>
                                    <td>${piece.id_piece}</td>
                                    <td>${piece.nom_piece}</td>
                                    <td>${piece.reference}</td>
                                    <td>${piece.nom_type}</td>
                                    <td>${piece.nom_rayon}</td>
                                    <td>${piece.stock_initial}</td>
                                    <td>${new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'MGA' }).format(piece.prix_unitaire)}</td>
                                </tr>
                            `;
                            piecesInStockTableBody.insertAdjacentHTML('beforeend', row);
                        });
                    }
                    // Generate pagination links for the modal
                    generatePagination(piecesInStockPagination, data.current_page, data.total_pages, 'loadPiecesStockDetails');

                } else {
                    showMessage("Erreur lors du chargement des détails des pièces en stock : " + data.message, "error");
                }
            } catch (error) {
                console.error('Erreur AJAX:', error);
                showMessage("Erreur technique lors du chargement des détails des pièces en stock.", "error");
            }
        }

        async function loadPneusStockDetails(page = 1) {
            try {
                const response = await fetch(`admin.php?action=get_all_pneus_in_stock&page=${page}`);
                const data = await response.json();

                if (data.success) {
                    const pneusInStockTableBody = document.getElementById('pneusInStockTableBody');
                    const pneusInStockPagination = document.getElementById('pneusInStockPagination');
                    pneusInStockTableBody.innerHTML = ''; // Clear existing rows
                    if (data.data.length === 0) {
                        pneusInStockTableBody.innerHTML = '<tr><td colspan="8" class="text-center">Aucun modèle de pneu en stock.</td></tr>';
                    } else {
                        data.data.forEach(pneu => {
                            const row = `
                                <tr>
                                    <td>${pneu.id_pneu}</td>
                                    <td>${pneu.nom_modele}</td>
                                    <td>${pneu.nom_marque}</td>
                                    <td>${pneu.valeur_dimension}</td>
                                    <td>${pneu.nom_profil}</td>
                                    <td>${pneu.quantite_en_stock}</td>
                                    <td>${new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'MGA' }).format(pneu.prix_achat_ht)}</td>
                                    <td>${pneu.tva_applicable}%</td>
                                </tr>
                            `;
                            pneusInStockTableBody.insertAdjacentHTML('beforeend', row);
                        });
                    }
                    // Generate pagination links for the modal
                    generatePagination(pneusInStockPagination, data.current_page, data.total_pages, 'loadPneusStockDetails');

                } else {
                    showMessage("Erreur lors du chargement des détails des pneus en stock : " + data.message, "error");
                }
            } catch (error) {
                console.error('Erreur AJAX:', error);
                showMessage("Erreur technique lors du chargement des détails des pneus en stock.", "error");
            }
        }

        // --- Fonctions pour charger et afficher les données des tableaux du dashboard (AJAX) ---
        // Rendre ces fonctions globales pour être appelées par la pagination générique
        async function loadSortiesPiecesData(page = 1) {
            try {
                const response = await fetch(`admin.php?action=get_sorties_pieces_data&page=${page}`);
                const data = await response.json();
                if (data.success) {
                    const sortiesPiecesTableBody = document.getElementById('sortiesPiecesTableBody');
                    const sortiesPiecesPagination = document.getElementById('sortiesPiecesPagination');
                    sortiesPiecesTableBody.innerHTML = '';
                    if (data.data.length === 0) {
                        sortiesPiecesTableBody.innerHTML = '<tr><td colspan="6" class="text-center">Aucune sortie de pièce récente.</td></tr>';
                    } else {
                        data.data.forEach(sortie => {
                            const row = `
                                <tr>
                                    <td>${sortie.id_sortie}</td>
                                    <td>${new Date(sortie.date_sortie).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
                                    <td>${sortie.magasinier}</td>
                                    <td>${sortie.id_intervention}</td>
                                    <td>${sortie.vehicule_immat}</td>
                                    <td>${new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'MGA' }).format(sortie.total_ttc)}</td>
                                </tr>
                            `;
                            sortiesPiecesTableBody.insertAdjacentHTML('beforeend', row);
                        });
                    }
                    generatePagination(sortiesPiecesPagination, data.current_page, data.total_pages, 'loadSortiesPiecesData');
                } else {
                    showMessage("Erreur lors du chargement des sorties de pièces: " + data.message, "error");
                }
            } catch (error) {
                console.error('Erreur AJAX:', error);
                showMessage("Erreur technique lors du chargement des sorties de pièces.", "error");
            }
        }

        async function loadSortiesPneusData(page = 1) {
            try {
                const response = await fetch(`admin.php?action=get_sorties_pneus_data&page=${page}`);
                const data = await response.json();
                if (data.success) {
                    const sortiesPneusTableBody = document.getElementById('sortiesPneusTableBody');
                    const sortiesPneusPagination = document.getElementById('sortiesPneusPagination');
                    sortiesPneusTableBody.innerHTML = '';
                    if (data.data.length === 0) {
                        sortiesPneusTableBody.innerHTML = '<tr><td colspan="8" class="text-center">Aucune sortie de pneu récente.</td></tr>';
                    } else {
                        data.data.forEach(sortie_pneu => {
                            // Assure que la valeur est affichée même si mpv.position_pneu est null
                            const vehiculeInfo = sortie_pneu.position_pneu && sortie_pneu.immatriculation
                                ? `${sortie_pneu.position_pneu} (${sortie_pneu.immatriculation})`
                                : (sortie_pneu.immatriculation ? `(Véhicule: ${sortie_pneu.immatriculation})` : 'N/A');
                            const row = `
                                <tr>
                                    <td>${sortie_pneu.id_sortie_pneu}</td>
                                    <td>${new Date(sortie_pneu.date_sortie).toLocaleString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
                                    <td>${sortie_pneu.magasinier}</td>
                                    <td>${sortie_pneu.numero_serie}</td>
                                    <td>${sortie_pneu.nom_modele}</td>
                                    <td>${vehiculeInfo}</td>
                                    <td>${sortie_pneu.statut_final}</td>
                                    <td>${sortie_pneu.raison_sortie ?? 'N/A'}</td>
                                </tr>
                            `;
                            sortiesPneusTableBody.insertAdjacentHTML('beforeend', row);
                        });
                    }
                    generatePagination(sortiesPneusPagination, data.current_page, data.total_pages, 'loadSortiesPneusData');
                } else {
                    showMessage("Erreur lors du chargement des sorties de pneus: " + data.message, "error");
                }
            } catch (error) {
                console.error('Erreur AJAX:', error);
                showMessage("Erreur technique lors du chargement des sorties de pneus.", "error");
            }
        }

        async function loadAlertesPiecesData(page = 1) {
            try {
                const response = await fetch(`admin.php?action=get_alertes_pieces_data&page=${page}`);
                const data = await response.json();
                if (data.success) {
                    const alertesPiecesTableBody = document.getElementById('alertesPiecesTableBody');
                    const alertesPiecesPagination = document.getElementById('alertesPiecesPagination');
                    alertesPiecesTableBody.innerHTML = '';
                    if (data.data.length === 0) {
                        alertesPiecesTableBody.innerHTML = '<tr><td colspan="5" class="text-green-400 text-center">Aucune alerte de stock bas pour les pièces.</td></tr>';
                    } else {
                        data.data.forEach(piece => {
                            const row = `
                                <tr class="low-stock-row">
                                    <td>${piece.nom_piece}</td>
                                    <td>${piece.reference}</td>
                                    <td>${piece.nom_type}</td>
                                    <td>${piece.stock_initial}</td>
                                    <td>${piece.seuil_alerte}</td>
                                </tr>
                            `;
                            alertesPiecesTableBody.insertAdjacentHTML('beforeend', row);
                        });
                    }
                    generatePagination(alertesPiecesPagination, data.current_page, data.total_pages, 'loadAlertesPiecesData');
                } else {
                    showMessage("Erreur lors du chargement des alertes pièces: " + data.message, "error");
                }
            } catch (error) {
                console.error('Erreur AJAX:', error);
                showMessage("Erreur technique lors du chargement des alertes pièces.", "error");
            }
        }

        async function loadAlertesPneusData(page = 1) {
            try {
                const response = await fetch(`admin.php?action=get_alertes_pneus_data&page=${page}`);
                const data = await response.json();
                if (data.success) {
                    const alertesPneusTableBody = document.getElementById('alertesPneusTableBody');
                    const alertesPneusPagination = document.getElementById('alertesPneusPagination');
                    alertesPneusTableBody.innerHTML = '';
                    if (data.data.length === 0) {
                        alertesPneusTableBody.innerHTML = '<tr><td colspan="5" class="text-green-400 text-center">Aucune alerte de stock bas pour les pneus.</td></tr>';
                    } else {
                        data.data.forEach(pneu => {
                            const row = `
                                <tr class="low-stock-row">
                                    <td>${pneu.nom_modele}</td>
                                    <td>${pneu.nom_marque}</td>
                                    <td>${pneu.valeur_dimension}</td>
                                    <td>${pneu.nom_profil}</td>
                                    <td>${pneu.quantiter_initial}</td>
                                </tr>
                            `;
                            alertesPneusTableBody.insertAdjacentHTML('beforeend', row);
                        });
                    }
                    generatePagination(alertesPneusPagination, data.current_page, data.total_pages, 'loadAlertesPneusData');
                } else {
                    showMessage("Erreur lors du chargement des alertes pneus: " + data.message, "error");
                }
            } catch (error) {
                console.error('Erreur AJAX:', error);
                showMessage("Erreur technique lors du chargement des alertes pneus.", "error");
            }
        }


        // Generic pagination generator function (for both modals and dashboard tables)
        function generatePagination(container, currentPage, totalPages, loadFunction) {
            container.innerHTML = ''; // Clear previous pagination
            if (totalPages <= 1) return;

            // Previous button
            const prevClass = currentPage === 1 ? 'disabled' : '';
            const prevLink = `<a href="#" class="${prevClass}" data-page="${currentPage - 1}">Précédent</a>`;
            container.insertAdjacentHTML('beforeend', prevLink);

            // Page numbers
            const maxPagesToShow = 5; // Max number of page buttons to show
            let startPage = Math.max(1, currentPage - Math.floor(maxPagesToShow / 2));
            let endPage = Math.min(totalPages, startPage + maxPagesToShow - 1);

            // Adjust startPage if endPage reached totalPages and we have less than maxPagesToShow
            if (endPage - startPage + 1 < maxPagesToShow) {
                startPage = Math.max(1, endPage - maxPagesToShow + 1);
            }

            if (startPage > 1) {
                container.insertAdjacentHTML('beforeend', `<a href="#" data-page="1">1</a>`);
                if (startPage > 2) {
                    container.insertAdjacentHTML('beforeend', `<span>...</span>`);
                }
            }

            for (let i = startPage; i <= endPage; i++) {
                const pageClass = i === currentPage ? 'current-page' : '';
                const pageLink = `<a href="#" class="${pageClass}" data-page="${i}">${i}</a>`;
                container.insertAdjacentHTML('beforeend', pageLink);
            }

            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    container.insertAdjacentHTML('beforeend', `<span>...</span>`);
                }
                container.insertAdjacentHTML('beforeend', `<a href="#" data-page="${totalPages}">${totalPages}</a>`);
            }


            // Next button
            const nextClass = currentPage === totalPages ? 'disabled' : '';
            const nextLink = `<a href="#" class="${nextClass}" data-page="${currentPage + 1}">Suivant</a>`;
            container.insertAdjacentHTML('beforeend', nextLink);

            // Add event listeners to new links
            container.querySelectorAll('a').forEach(link => {
                if (!link.classList.contains('disabled')) {
                    link.addEventListener('click', (e) => {
                        e.preventDefault();
                        const page = parseInt(e.target.dataset.page);
                        // Call the appropriate load function from the global scope
                        window[loadFunction](page);
                    });
                }
            });
        }


        document.addEventListener('DOMContentLoaded', function() {
            const burgerMenu = document.querySelector('.burger-menu');
            const navbarNav = document.querySelector('.navbar-nav');
            const dropdownToggle = document.querySelector('.dropdown-toggle');

            // KPI Cards for dynamic details
            const kpiPiecesStockCard = document.getElementById('kpi-pieces-stock');
            const kpiPneusStockCard = document.getElementById('kpi-pneus-stock');

            // Modals for details
            const piecesInStockModal = document.getElementById('piecesInStockModal');
            const closePiecesInStockModalBtn = document.getElementById('closePiecesInStockModal');
            // const piecesInStockTableBody = document.getElementById('piecesInStockTableBody'); // Now global
            // const piecesInStockPagination = document.getElementById('piecesInStockPagination'); // Now global

            const pneusInStockModal = document.getElementById('pneusInStockModal');
            const closePneusInStockModalBtn = document.getElementById('closePneusInStockModal');
            // const pneusInStockTableBody = document.getElementById('pneusInStockTableBody'); // Now global
            // const pneusInStockPagination = document.getElementById('pneusInStockPagination'); // Now global

            // Dashboard table elements for AJAX pagination
            // const sortiesPiecesTableBody = document.getElementById('sortiesPiecesTableBody'); // Now global
            // const sortiesPiecesPagination = document.getElementById('sortiesPiecesPagination'); // Now global
            // const sortiesPneusTableBody = document.getElementById('sortiesPneusTableBody'); // Now global
            // const sortiesPneusPagination = document.getElementById('sortiesPneusPagination'); // Now global
            // const alertesPiecesTableBody = document.getElementById('alertesPiecesTableBody'); // Now global
            // const alertesPiecesPagination = document.getElementById('alertesPiecesPagination'); // Now global
            // const alertesPneusTableBody = document.getElementById('alertesPneusTableBody'); // Now global
            // const alertesPneusPagination = document.getElementById('alertesPneusPagination'); // Now global


            // Gère l'ouverture/fermeture du menu burger
            burgerMenu.addEventListener('click', () => {
                navbarNav.classList.toggle('open');
                burgerMenu.classList.toggle('open');
            });

            // Gère l'affichage du menu déroulant (dropdown) au CLIC
            if (dropdownToggle) {
                dropdownToggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    this.closest('.dropdown').classList.toggle('active');
                    document.querySelectorAll('.dropdown.active').forEach(otherDropdown => {
                        if (otherDropdown !== this.closest('.dropdown')) {
                            otherDropdown.classList.remove('active');
                        }
                    });
                });
            }

            // Fermer le dropdown si l'on clique en dehors
            document.addEventListener('click', function(event) {
                const dropdown = document.querySelector('.navbar-nav .dropdown');
                if (dropdown && !dropdown.contains(event.target) && event.target !== dropdownToggle) {
                    dropdown.classList.remove('active');
                    if (navbarNav.classList.contains('open') && window.innerWidth <= 768) {
                        navbarNav.classList.remove('open');
                        burgerMenu.classList.remove('open');
                    }
                }
            });
            
            // --- Gestion du lien actif dans la navigation ---
            function setActiveLink() {
                const currentFilename = window.location.pathname.split('/').pop();
                document.querySelectorAll('.navbar-nav a').forEach(link => {
                    const linkFilename = link.getAttribute('href').split('/').pop();
                    // Assurez-vous que les liens du dashboard et de la gestion pneumatique sont correctement gérés
                    if (currentFilename === 'admin.php' && linkFilename === 'admin.php') {
                        link.classList.add('active');
                    } else if (currentFilename === linkFilename && currentFilename !== 'admin.php') {
                         link.classList.add('active');
                         const parentDropdown = link.closest('.dropdown');
                        if (parentDropdown) {
                            parentDropdown.classList.add('active');
                        }
                    } else {
                        link.classList.remove('active');
                    }
                });
            }
            setActiveLink();

            // Initialisation du graphique Chart.js
            const ctx = document.getElementById('caPiecesChart').getContext('2d');
            const caPiecesChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: <?php echo json_encode($chart_labels); ?>,
                    datasets: [{
                        label: 'CA Pièces (TTC) par Mois',
                        data: <?php echo json_encode($chart_data); ?>,
                        borderColor: 'rgb(255, 218, 71)', // var(--accent-color)
                        backgroundColor: 'rgba(255, 218, 71, 0.2)',
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            labels: {
                                color: 'var(--text-color)'
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return context.dataset.label + ': ' + new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'MGA' }).format(context.raw);
                                }
                            }
                        }
                    },
                    scales: {
                        x: {
                            ticks: {
                                color: 'var(--text-color)'
                            },
                            grid: {
                                color: 'rgba(255, 255, 255, 0.1)'
                            }
                        },
                        y: {
                            ticks: {
                                color: 'var(--text-color)',
                                callback: function(value) {
                                    return new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'MGA', minimumFractionDigits: 0, maximumFractionDigits: 0 }).format(value);
                                }
                            },
                            grid: {
                                color: 'rgba(255, 255, 255, 0.1)'
                            }
                        }
                    }
                }
            });


            // --- Event Listeners for KPI Cards to open modals ---

            kpiPiecesStockCard.addEventListener('click', () => {
                piecesInStockModal.classList.add('open');
                loadPiecesStockDetails(1); // Load first page of data
            });

            closePiecesInStockModalBtn.addEventListener('click', () => {
                piecesInStockModal.classList.remove('open');
            });

            window.addEventListener('click', (event) => {
                if (event.target === piecesInStockModal) {
                    piecesInStockModal.classList.remove('open');
                }
            });

            kpiPneusStockCard.addEventListener('click', () => {
                pneusInStockModal.classList.add('open');
                loadPneusStockDetails(1); // Load first page of data
            });

            closePneusInStockModalBtn.addEventListener('click', () => {
                pneusInStockModal.classList.remove('open');
            });

            window.addEventListener('click', (event) => {
                if (event.target === pneusInStockModal) {
                    pneusInStockModal.classList.remove('open');
                }
            });

            // Initial load for dashboard tables when the page loads
            loadSortiesPiecesData(1);
            loadSortiesPneusData(1);
            loadAlertesPiecesData(1);
            loadAlertesPneusData(1);

        });

        // Génération des étincelles (sparkles) - tiré des pages précédentes
        const sparklesContainer = document.querySelector('.sparkles-container');
        function createSparkle() {
            const sparkle = document.createElement('div');
            sparkle.classList.add('spark');
            const size = Math.random() * 3 + 2; // Taille entre 2px et 5px
            sparkle.style.width = `${size}px`;
            sparkle.style.height = `${size}px`;
            sparkle.style.left = `${Math.random() * 100}vw`;
            sparkle.style.top = `${-10 - Math.random() * 10}px`; // Commencer au-dessus de l'écran
            sparkle.style.setProperty('--random-x', `${(Math.random() - 0.5) * 200}px`); // Mouvement horizontal aléatoire
            sparkle.style.animationDuration = `${Math.random() * 1.5 + 1.5}s`; // Durée entre 1.5s et 3s
            sparkle.style.animationDelay = `${Math.random() * 2}s`; // Délai avant de commencer

            sparklesContainer.appendChild(sparkle);

            // Supprimer l'étincelle après son animation pour éviter l'accumulation
            sparkle.addEventListener('animationend', () => {
                sparkle.remove();
            });
        }
        setInterval(createSparkle, 200); // Créer une étincelle toutes les 200ms
    </script>
</body>
</html>
