<?php
// achats_effectues.php
session_start();
require_once 'php/config.php';
date_default_timezone_set('Indian/Antananarivo');

if (
    !isset($_SESSION['loggedin']) 
    || $_SESSION['loggedin'] !== true 
    || !isset($_SESSION['role']) 
    || $_SESSION['role'] !== 'Administrateur'
) {
    header('Location: index');
    exit();
}


$pdo = getDbConnection();

// =====================================================================
// Helpers filtres pour Achats
// =====================================================================
function buildAchatsFilters(array $src, &$binds) {
    $w = [];
    $binds = [];

    // type: all | pieces | pneus
    if (!empty($src['type']) && in_array($src['type'], ['pieces','pneus'], true)) {
        $w[] = "category = ?";
        $binds[] = $src['type'];
    }

    if (!empty($src['date_from'])) {
        $w[] = "date_achat >= ?";
        $binds[] = $src['date_from'] . (strlen($src['date_from']) === 10 ? " 00:00:00" : "");
    }

    if (!empty($src['date_to'])) {
        $w[] = "date_achat <= ?";
        $binds[] = $src['date_to'] . (strlen($src['date_to']) === 10 ? " 23:59:59" : "");
    }

    if (!empty($src['q'])) {
        $w[] = "article LIKE ?";
        $binds[] = "%".$src['q']."%";
    }

    return $w ? ('WHERE '.implode(' AND ', $w)) : '';
}

// =====================================================================
// Endpoint AJAX: Achats (JSON paginé)
// =====================================================================
if (isset($_GET['action']) && $_GET['action'] === 'get_achats') {
    header('Content-Type: application/json; charset=UTF-8');

    $perPage = (isset($_GET['perPage']) && is_numeric($_GET['perPage'])) ? max(5, (int)$_GET['perPage']) : 10;
    $page = (isset($_GET['page']) && is_numeric($_GET['page'])) ? max(1, (int)$_GET['page']) : 1;
    $offset = ($page - 1) * $perPage;

    // CTE unifiée pour achats pièces + pneus
    $sqlBase = "
        WITH achats_pieces AS (
            SELECT
                'pieces' AS category,
                r.date_reception AS date_achat,
                p.nom_piece AS article,
                lr.quantite_receptionnee AS qte,
                lr.prix_achat_ht_reception AS pu_ht,
                lr.tva_reception AS tva_pct,
                (lr.quantite_receptionnee * lr.prix_achat_ht_reception) AS montant_ht,
                (lr.quantite_receptionnee * lr.prix_achat_ht_reception * (lr.tva_reception/100)) AS montant_tva,
                (lr.quantite_receptionnee * lr.prix_achat_ht_reception * (1 + lr.tva_reception/100)) AS montant_ttc
            FROM lignes_reception lr
            JOIN receptions r ON lr.id_reception = r.id_reception
            JOIN piece p ON lr.id_piece = p.id_piece
        ),
        achats_pneus AS (
            SELECT
                'pneus' AS category,
                rp.date_reception AS date_achat,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                lrp.quantite_receptionnee AS qte,
                pn.prix_achat_ht AS pu_ht,
                pn.tva_applicable AS tva_pct,
                (lrp.quantite_receptionnee * pn.prix_achat_ht) AS montant_ht,
                (lrp.quantite_receptionnee * pn.prix_achat_ht * (pn.tva_applicable/100)) AS montant_tva,
                (lrp.quantite_receptionnee * pn.prix_achat_ht * (1 + pn.tva_applicable/100)) AS montant_ttc
            FROM lignes_reception_pneu lrp
            JOIN receptions_pneu rp ON lrp.id_reception_pneu = rp.id_reception_pneu
            JOIN pneus pn ON lrp.id_pneu = pn.id_pneu
        ),
        achats AS (
            SELECT * FROM achats_pieces
            UNION ALL
            SELECT * FROM achats_pneus
        ),
        achats_filtres AS (
            SELECT * FROM achats
            /*WHERE_INLINE*/
        )
        SELECT
            category, date_achat, article, qte, pu_ht, tva_pct,
            montant_ht, montant_tva, montant_ttc
        FROM achats_filtres
    ";

    $binds = [];
    $where = buildAchatsFilters($_GET, $binds);
    $sqlBaseInjected = str_replace('/*WHERE_INLINE*/', $where, $sqlBase);

    // total count
    $sqlCount = "SELECT COUNT(*) FROM ($sqlBaseInjected) t";
    $stmt = $pdo->prepare($sqlCount);
    $stmt->execute($binds);
    $total = (int)$stmt->fetchColumn();

    // data page
    $sqlData = "SELECT * FROM ($sqlBaseInjected) t ORDER BY date_achat DESC, article ASC LIMIT ? OFFSET ?";
    $stmt = $pdo->prepare($sqlData);
    $execBinds = $binds;
    $execBinds[] = $perPage;
    $execBinds[] = $offset;
    $stmt->execute($execBinds);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true,
        'data' => $rows,
        'current_page' => $page,
        'total_pages' => max(1, (int)ceil($total / $perPage)),
        'total_records' => $total
    ]);
    exit();
}

if (isset($_GET['action']) && $_GET['action'] === 'export_achats') {
    $binds = [];
    $where = buildAchatsFilters($_GET, $binds);

    $sql = "
        WITH achats_pieces AS (
            SELECT
                'pieces' AS category,
                r.date_reception AS date_achat,
                p.nom_piece AS article,
                lr.quantite_receptionnee AS qte,
                lr.prix_achat_ht_reception AS pu_ht,
                lr.tva_reception AS tva_pct,
                (lr.quantite_receptionnee * lr.prix_achat_ht_reception) AS montant_ht,
                (lr.quantite_receptionnee * lr.prix_achat_ht_reception * (lr.tva_reception/100)) AS montant_tva,
                (lr.quantite_receptionnee * lr.prix_achat_ht_reception * (1 + lr.tva_reception/100)) AS montant_ttc
            FROM lignes_reception lr
            JOIN receptions r ON lr.id_reception = r.id_reception
            JOIN piece p ON lr.id_piece = p.id_piece
        ),
        achats_pneus AS (
            SELECT
                'pneus' AS category,
                rp.date_reception AS date_achat,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                lrp.quantite_receptionnee AS qte,
                pn.prix_achat_ht AS pu_ht,
                pn.tva_applicable AS tva_pct,
                (lrp.quantite_receptionnee * pn.prix_achat_ht) AS montant_ht,
                (lrp.quantite_receptionnee * pn.prix_achat_ht * (pn.tva_applicable/100)) AS montant_tva,
                (lrp.quantite_receptionnee * pn.prix_achat_ht * (1 + pn.tva_applicable/100)) AS montant_ttc
            FROM lignes_reception_pneu lrp
            JOIN receptions_pneu rp ON lrp.id_reception_pneu = rp.id_reception_pneu
            JOIN pneus pn ON lrp.id_pneu = pn.id_pneu
        ),
        achats AS (
            SELECT * FROM achats_pieces
            UNION ALL
            SELECT * FROM achats_pneus
        ),
        achats_filtres AS (
            SELECT * FROM achats
            $where
        )
        SELECT category, date_achat, article, qte, pu_ht, tva_pct, montant_ht, montant_tva, montant_ttc
        FROM achats_filtres
        ORDER BY date_achat DESC, article ASC
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($binds);

    $filename = 'achats_'.date('Ymd_His').'.csv';
    header('Content-Type: text/csv; charset=UTF-8');
    header("Content-Disposition: attachment; filename=\"$filename\"");
    header('Pragma: no-cache');
    header('Expires: 0');

    // BOM UTF-8 pour Excel
    echo "\xEF\xBB\xBF";

    $out = fopen('php://output', 'w');
    fputcsv($out, ['Catégorie','Date réception','Article','Quantité','PU HT','TVA %','Montant HT','Montant TVA','Montant TTC'], ';');

    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        fputcsv($out, [
            $r['category'],
            $r['date_achat'],
            $r['article'],
            number_format($r['qte'], 0, ',', ' '),
            number_format($r['pu_ht'], 2, ',', ' '),
            number_format($r['tva_pct'], 2, ',', ' '),
            number_format($r['montant_ht'], 2, ',', ' '),
            number_format($r['montant_tva'], 2, ',', ' '),
            number_format($r['montant_ttc'], 2, ',', ' ')
        ], ';');
    }
    fclose($out);
    exit();
}


// =====================================================================
// Endpoint AJAX: Totaux HT & TTC filtrés
// =====================================================================
if (isset($_GET['action']) && $_GET['action'] === 'get_totaux') {
    header('Content-Type: application/json; charset=UTF-8');

    $binds = [];
    $where = buildAchatsFilters($_GET, $binds);

    $sql = "
        WITH achats_pieces AS (
            SELECT
                'pieces' AS category,
                r.date_reception AS date_achat,
                p.nom_piece AS article,
                (lr.quantite_receptionnee * lr.prix_achat_ht_reception) AS montant_ht,
                (lr.quantite_receptionnee * lr.prix_achat_ht_reception * (1 + lr.tva_reception/100)) AS montant_ttc
            FROM lignes_reception lr
            JOIN receptions r ON lr.id_reception = r.id_reception
            JOIN piece p ON lr.id_piece = p.id_piece
        ),
        achats_pneus AS (
            SELECT
                'pneus' AS category,
                rp.date_reception AS date_achat,
                CONCAT('Pneu: ', pn.nom_modele) AS article,
                (lrp.quantite_receptionnee * pn.prix_achat_ht) AS montant_ht,
                (lrp.quantite_receptionnee * pn.prix_achat_ht * (1 + pn.tva_applicable/100)) AS montant_ttc
            FROM lignes_reception_pneu lrp
            JOIN receptions_pneu rp ON lrp.id_reception_pneu = rp.id_reception_pneu
            JOIN pneus pn ON lrp.id_pneu = pn.id_pneu
        ),
        achats AS (
            SELECT * FROM achats_pieces
            UNION ALL
            SELECT * FROM achats_pneus
        ),
        achats_filtres AS (
            SELECT * FROM achats
            $where
        )
        SELECT
            COALESCE(SUM(montant_ht),0) AS total_ht,
            COALESCE(SUM(montant_ttc),0) AS total_ttc
        FROM achats_filtres
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($binds);
    $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: ['total_ht'=>0,'total_ttc'=>0];

    echo json_encode([
        'success' => true,
        'total_ht' => (float)$row['total_ht'],
        'total_ttc' => (float)$row['total_ttc'],
    ]);
    exit();
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Achats effectués</title>
    <link rel="icon" type="image/png" href="img/logo_ase.png">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <script src="https://cdn.tailwindcss.com"></script>
<style>
:root {
    --gradient-bg: linear-gradient(135deg, #ffffff 65%, #2ecc71 100%);
    --text-color: #222;
    --heading-color: #002147;
    --accent-color: #2ecc71;
    --border-color: rgba(0,0,0,0.1);
    --input-bg: rgba(255, 255, 255, 0.95);
    --input-border: #ccc;
    --table-row-even-bg: rgba(255,255,255,0.5);
}
body { margin:0; font-family:'Inter',sans-serif; background:var(--gradient-bg); background-attachment:fixed; color:var(--text-color); }
.navbar { display:flex; justify-content:space-between; align-items:center; background:var(--gradient-bg); backdrop-filter: blur(8px); padding:15px 30px; position:sticky; top:0; z-index:10; }
.navbar img { height:45px; }
.navbar-nav { list-style:none; display:flex; gap:24px; }
.navbar-nav a { color:var(--heading-color); text-decoration:none; display:flex; align-items:center; gap:8px; font-weight:500; }
.navbar-nav a:hover, .navbar-nav a.active { color:var(--accent-color); }
.main { max-width:2000px; width:100%; margin:30px auto; background:var(--gradient-bg); backdrop-filter: blur(6px); padding:30px; border-radius:10px; }
h1 { color:var(--heading-color); font-size:2.0rem; margin:0 0 10px; text-align:center;}
.section { background:var(--gradient-bg); border-radius:10px; padding:20px; margin-top:20px; backdrop-filter:blur(4px); color:black; }
.filters { display:grid; grid-template-columns: repeat(auto-fit,minmax(200px,1fr)); gap:12px; margin-bottom:12px; }
.input, select { background: var(--input-bg); border:1px solid var(--input-border); border-radius:8px; padding:10px 12px; color:var(--text-color); width:100%; }

/* Modern buttons */
.btn {
    background: linear-gradient(135deg, #2563eb, #1d4ed8);
    color: #fff;
    border: none;
    border-radius: 6px;
    padding: 10px 16px;
    cursor: pointer;
    font-weight: 100;
    box-shadow: 0 2px 4px rgba(0,0,0,0.08);
    transition: all 0.2s ease;
}
.btn:hover {
    background: linear-gradient(135deg, #1d4ed8, #2563eb);
    transform: translateY(-1px);
}
.btn-outline {
    background: transparent;
    border: 2px solid var(--accent-color);
    color: var(--accent-color);
    border-radius: 6px;
    padding: 10px 16px;
    font-weight: 500;
    transition: all 0.2s ease;
}
.btn-outline:hover {
    background: var(--accent-color);
    color: #000;
    transform: translateY(-1px);
}

/* Totals cards */
.kpi-row { display:flex; gap:16px; flex-wrap:wrap; justify-content:center; margin-bottom: 8px; }
.kpi-card {
    background:#fff; color:#111; border-radius:10px; padding:16px 20px; min-width:260px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
}
.kpi-card .kpi-title { font-size:0.95rem; color:#555; }
.kpi-card .kpi-value { font-size:1.6rem; font-weight:800; color:var(--heading-color); margin-top:6px; }
.kpi-card .kpi-sub { font-size:0.8rem; color:#888; }

.table-wrap { overflow:auto; border:1px solid var(--border-color); border-radius:10px; }
table { width:100%; border-collapse:collapse; min-width:1100px; }
th, td { border-bottom:1px solid var(--input-border); padding:10px; text-align:left; font-size:.9rem; }
th { background: rgba(0,33,71,0.85); color:#fff; text-transform:uppercase; font-size:.8rem; letter-spacing:.03em; position:sticky; top:0; }
tr:nth-child(even) { background: var(--table-row-even-bg); }

/* Pagination améliorée */
.pagination {
    display: flex;
    gap: 6px;
    justify-content: center;
    align-items: center;
    margin-top: 16px;
    flex-wrap: wrap;
}
.pagination button,
.pagination a,
.pagination span {
    min-width: 36px;
    padding: 8px 12px;
    border-radius: 6px;
    border: 1px solid var(--input-border);
    background: var(--input-bg);
    color: var(--text-color);
    font-size: 0.9rem;
    cursor: pointer;
    transition: all 0.2s ease;
    text-decoration: none;
}
.pagination a:hover,
.pagination button:hover { background: var(--accent-color); color: #000; }
.pagination .current {
    background: var(--accent-color);
    color: #000;
    font-weight: 700;
    border-color: var(--accent-color);
}
.pagination button:disabled { opacity: 0.5; cursor: not-allowed; }

@media (max-width: 768px) { .navbar-nav { display:none; } .main { width:96%; } }
</style>
</head>
<body>
    <nav class="navbar">
        <div><img src="img/logo_ase.png" alt="Logo"></div>
        <ul class="navbar-nav">
            <li><a href="tableau_bord"><i class="fas fa-tachometer-alt"></i> Tableau de bord</a></li>
            <li><a href="details_sorties_globales"><i class="fas fa-gauge-high"></i> Admin legacy</a></li>
            <li><a href="rapports_analyses"><i class="fas fa-chart-pie"></i> Rapports</a></li>
            <li><a href="carnet_entretien_vehicule"><i class="fas fa-truck"></i> Flotte</a></li>
            <li><a href="rapport_pneumatique"><i class="fas fa-history"></i> Détails pneumatique</a></li>
            <li><a class="active" href="achats_effectues"><i class="fas fa-boxes-packing"></i> Achats effectués</a></li>
            <li><a href="sorties_effectuees"><i class="fas fa-truck-loading"></i> Sorties effectuées</a></li>
            <li><a href="php/authentification.php?action=logout"><i class="fa-solid fa-power-off"></i></a></li>
        </ul>
    </nav>


    <main class="main">
        <h1><i class="fas fa-boxes-stacked"></i> <strong>Achats effectués (réceptions)</strong></h1>

        <!-- Totaux -->
        <div class="kpi-row">
            <div class="kpi-card">
                <div class="kpi-title">Total HT (filtré)</div>
                <div id="totalHtCard" class="kpi-value">0,00 Ar</div>
            </div>
            <div class="kpi-card">
                <div class="kpi-title">Total TTC (filtré)</div>
                <div id="totalTtcCard" class="kpi-value">0,00 Ar</div>
            </div>
        </div>

        <!-- Filtres -->
        <section class="section">
            <h2 class="text-xl mb-3" style="color:var(--heading-color)">Filtres</h2>
            <div class="filters">
                <div>
                    <label class="block text-sm mb-1">Type</label>
                    <select id="filter_type" class="input" style="background-color:white; color:black;">
                        <option value="">Tous</option>
                        <option value="pieces">Pièces</option>
                        <option value="pneus">Pneus</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm mb-1">Date début</label>
                    <input type="date" id="filter_date_from" class="input" style="background-color:white; color:black;">
                </div>
                <div>
                    <label class="block text-sm mb-1">Date fin</label>
                    <input type="date" id="filter_date_to" class="input" style="background-color:white; color:black;">
                </div>
                <div>
                    <label class="block text-sm mb-1">Recherche</label>
                    <input type="text" id="filter_q" placeholder="Article..." class="input" style="background-color:white; color:black;">
                </div>
                <div class="flex items-end gap-2">
                    <button class="btn" id="btnApplyFilters"><i class="fas fa-filter"></i>Filter</button>
                    <button class="btn btn-outline" id="btnResetFilters" type="button"><i class="fas fa-undo"></i>Reset</button>
                    <button class="btn" id="btnExport"><i class="fas fa-file-excel"></i>CSV</button>
                </div>
            </div>
        </section>

        <!-- Tableau -->
        <section class="section">
            <h2 class="text-xl mb-3" style="color:var(--heading-color)">Liste des achats</h2>
            <div class="table-wrap">
                <table id="achatsTable">
                    <thead>
                        <tr>
                            <th>Catégorie</th>
                            <th>Date réception</th>
                            <th>Article</th>
                            <th>Quantité</th>
                            <th>PU HT</th>
                            <th>TVA %</th>
                            <th>Montant HT</th>
                            <th>Montant TVA</th>
                            <th>Montant TTC</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr><td colspan="9">Chargement...</td></tr>
                    </tbody>
                </table>
            </div>
            <div class="pagination" id="pagination"></div>
        </section>
    </main>

<script>
const perPage = 10;

function paramsFromFilters(extra = {}) {
    const p = new URLSearchParams();
    p.set('type', document.getElementById('filter_type').value);
    p.set('date_from', document.getElementById('filter_date_from').value);
    p.set('date_to', document.getElementById('filter_date_to').value);
    p.set('q', document.getElementById('filter_q').value);
    p.set('perPage', perPage);
    for (const [k,v] of Object.entries(extra)) p.set(k, v);
    return p;
}

async function loadAchats(page = 1) {
    const tbody = document.querySelector('#achatsTable tbody');
    const pagDiv = document.getElementById('pagination');
    tbody.innerHTML = '<tr><td colspan="9">Chargement...</td></tr>';
    pagDiv.innerHTML = '';

    const params = paramsFromFilters({ action: 'get_achats', page });
    const url = window.location.pathname + '?' + params.toString();

    try {
        const res = await fetch(url);
        const json = await res.json();
        if (!json.success) throw new Error('Chargement échoué');

        if (json.data.length === 0) {
            tbody.innerHTML = '<tr><td colspan="9">Aucun achat trouvé.</td></tr>';
        } else {
            tbody.innerHTML = '';
            json.data.forEach(row => {
                const tr = document.createElement('tr');
                tr.innerHTML = `
                    <td>${row.category}</td>
                    <td>${row.date_achat}</td>
                    <td>${row.article}</td>
                    <td>${Number(row.qte).toLocaleString()}</td>
                    <td>${Number(row.pu_ht).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                    <td>${Number(row.tva_pct).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                    <td>${Number(row.montant_ht).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                    <td>${Number(row.montant_tva).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                    <td>${Number(row.montant_ttc).toLocaleString(undefined,{minimumFractionDigits:2})}</td>
                `;
                tbody.appendChild(tr);
            });
        }

        renderPagination(json.current_page, json.total_pages);
        loadTotaux(); // met à jour les totaux HT/TTC

    } catch (e) {
        tbody.innerHTML = `<tr><td colspan="9">Erreur: ${e.message}</td></tr>`;
    }
}

function renderPagination(current, total) {
    const pagDiv = document.getElementById('pagination');
    pagDiv.innerHTML = '';
    if (total <= 1) return;

    const addBtn = (label, page, disabled = false, isCurrent = false) => {
        if (isCurrent) {
            const span = document.createElement('span');
            span.className = 'current';
            span.textContent = label;
            pagDiv.appendChild(span);
        } else {
            const btn = document.createElement('button');
            btn.textContent = label;
            btn.disabled = disabled;
            if (!disabled) btn.onclick = () => loadAchats(page);
            pagDiv.appendChild(btn);
        }
    };

    addBtn('« Précédent', current - 1, current === 1);

    const delta = 2;
    let start = Math.max(1, current - delta);
    let end = Math.min(total, current + delta);

    if (start > 1) {
        addBtn('1', 1);
        if (start > 2) {
            const span = document.createElement('span');
            span.textContent = '...';
            pagDiv.appendChild(span);
        }
    }

    for (let p = start; p <= end; p++) {
        addBtn(String(p), p, false, p === current);
    }

    if (end < total) {
        if (end < total - 1) {
            const span = document.createElement('span');
            span.textContent = '...';
            pagDiv.appendChild(span);
        }
        addBtn(String(total), total);
    }

    addBtn('Suivant »', current + 1, current === total);
}

async function loadTotaux() {
    const params = paramsFromFilters({ action: 'get_totaux' });
    const url = window.location.pathname + '?' + params.toString();
    try {
        const res = await fetch(url);
        const json = await res.json();
        if (json.success) {
            document.getElementById('totalHtCard').textContent =
                Number(json.total_ht).toLocaleString(undefined,{minimumFractionDigits:2}) + ' Ar';
            document.getElementById('totalTtcCard').textContent =
                Number(json.total_ttc).toLocaleString(undefined,{minimumFractionDigits:2}) + ' Ar';
        }
    } catch(e) {
        console.error("Erreur totaux:", e);
    }
}

document.getElementById('btnApplyFilters').addEventListener('click', () => loadAchats(1));
document.getElementById('btnResetFilters').addEventListener('click', () => {
    document.getElementById('filter_type').value = '';
    document.getElementById('filter_date_from').value = '';
    document.getElementById('filter_date_to').value = '';
    document.getElementById('filter_q').value = '';
    loadAchats(1);
});

document.getElementById('btnExport').addEventListener('click', () => {
    const params = paramsFromFilters({ action: 'export_achats' });
    window.location = window.location.pathname + '?' + params.toString();
});

// Chargement initial au démarrage de la page
loadAchats(1);
loadTotaux();
</script>
</body>
</html>
